<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Cria a página de Configurações Globais para o Hub
 */
class WPCM_Hub_Settings {

    private $options;

    public function __construct() {
        $saved_options = get_option( 'wpcm_hub_global_settings' );
        if ( ! is_array( $saved_options ) ) { $saved_options = array(); }
        $defaults = self::get_default_options();
        $this->options = wp_parse_args( $saved_options, $defaults );

        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
    }

    /**
     * Contém todos os valores padrão
     */
    public static function get_default_options() {
        return array(
            'enable_banner'   => '1',
            'policy_page_url' => '',
            'policy_version'  => '1.0',
            'enable_iframe_blocker' => '1',
            'enable_ip_logging' => '0', 
            'enable_gcm_advanced' => '0', 
            
            'banner_position' => 'bottom-left',
            'bg_color'        => '#FFFFFF',
            'text_color'      => '#404040',
            'btn_accept_bg_color'    => '#4F46E5',
            'btn_reject_bg_color'    => 'transparent',
            'btn_reject_text_color'  => '#404040',
            'btn_reject_border_color' => '#CCCCCC',
            
            'banner_text'     => __( 'We use cookies on our website to give you the most relevant experience.', 'wpcm-agency-hub' ),
            'btn_accept_text' => __( 'Accept All', 'wpcm-agency-hub' ),
            'btn_reject_text' => __( 'Reject All', 'wpcm-agency-hub' ),
            'btn_settings_text' => __( 'Customize', 'wpcm-agency-hub' ),
            'modal_title'       => __( 'Manage Your Consent', 'wpcm-agency-hub' ),
            'btn_save_text'     => __( 'Save Preferences', 'wpcm-agency-hub' ),
            'iframe_blocker_text'   => __( 'This content is blocked. To view it, you must accept Marketing cookies.', 'wpcm-agency-hub' ),
            'iframe_blocker_button_text' => __( 'Accept & View Content', 'wpcm-agency-hub' ),
            'necessary_desc'    => __( 'These cookies are essential for the website to function properly.', 'wpcm-agency-hub' ),
            
            'agency_branding_name' => '',
            'agency_branding_url'  => '',
            'agency_whitelabel_name' => '',
            'agency_whitelabel_desc' => '',
            'agency_whitelabel_hide' => '0',
            'hub_log_retention_period' => '12',
        );
    }

    /**
     * Adiciona o submenu "Configurações Globais" ao Hub
     */
    public function add_settings_page() {
        add_submenu_page(
            'edit.php?post_type=wpcm_hub_site', 
            __( 'Global Settings', 'wpcm-agency-hub' ),
            __( 'Global Settings', 'wpcm-agency-hub' ),
            'manage_options',
            'wpcm_hub_settings',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Renderiza a página de abas (Design, Conteúdo)
     */
    public function render_settings_page() {
        global $wp_settings_sections;
        
        // CORREÇÃO: Adicionado phpcs:ignore pois 'tab' controla apenas a visualização da interface (navegação), não salva dados.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $active_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'design';
        ?>
        <div class="wrap">
            <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 10px;">
                <div>
                    <h1><?php esc_html_e( 'Global Consent Settings', 'wpcm-agency-hub' ); ?></h1>
                    <p><?php esc_html_e( 'These settings will be synced to all connected client sites.', 'wpcm-agency-hub' ); ?></p>
                </div>
                <div style="text-align: right; margin-top: 10px;">
                    <a href="#" id="wpcm-hub-sync-button" 
                       class="button button-primary button-large" 
                       data-nonce="<?php echo esc_attr( wp_create_nonce( 'wpcm_hub_sync_nonce' ) ); ?>"
                       title="<?php esc_attr_e( 'Force all sites to fetch new settings.', 'wpcm-agency-hub' ); ?>">
                        <?php esc_html_e( 'Sync All Sites', 'wpcm-agency-hub' ); ?>
                    </a>
                    <span class="spinner" style="float: none; vertical-align: middle; visibility: hidden; margin-left: 5px;"></span>
                    <p class="description"><?php esc_html_e( 'Force all sites to fetch new settings.', 'wpcm-agency-hub' ); ?></p>
                </div>
            </div>
            
            <div id="wpcm-sync-notice" class="notice" style="display: none; margin-left: 0;"></div>

            <h2 class="nav-tab-wrapper">
                <a href="?post_type=wpcm_hub_site&page=wpcm_hub_settings&tab=design" class="nav-tab <?php echo $active_tab == 'design' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Design', 'wpcm-agency-hub' ); ?></a>
                <a href="?post_type=wpcm_hub_site&page=wpcm_hub_settings&tab=content" class="nav-tab <?php echo $active_tab == 'content' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Content', 'wpcm-agency-hub' ); ?></a>
                <a href="?post_type=wpcm_hub_site&page=wpcm_hub_settings&tab=branding" class="nav-tab <?php echo $active_tab == 'branding' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Branding & Data', 'wpcm-agency-hub' ); ?></a>
            </h2>
            
            <form method="post" action="options.php">
                <?php
                settings_fields( 'wpcm_hub_settings_group' );
                $page_slug = 'wpcm_hub_settings_page_' . $active_tab;
                
                if ( ! isset( $wp_settings_sections[$page_slug] ) && $active_tab !== 'design' ) {
                     $page_slug = 'wpcm_hub_settings_page_design';
                     if ( $active_tab === 'content' ) $page_slug = 'wpcm_hub_settings_page_content';
                     if ( $active_tab === 'branding' ) $page_slug = 'wpcm_hub_settings_page_branding';
                }

                if ( isset( $wp_settings_sections[$page_slug] ) ) {
                    do_settings_sections( $page_slug );
                }
                
                submit_button( __( 'Save Global Settings', 'wpcm-agency-hub' ) );
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Registra os campos de configuração
     */
    public function register_settings() {
        register_setting(
            'wpcm_hub_settings_group',
            'wpcm_hub_global_settings',
            array( $this, 'sanitize_options' )
        );

        // --- Seção de Design ---
        $page_slug = 'wpcm_hub_settings_page_design';
        add_settings_section( 'wpcm_design_section', __( 'Banner Design', 'wpcm-agency-hub' ), null, $page_slug );
        add_settings_field( 'banner_position', __( 'Banner Position', 'wpcm-agency-hub' ), array( $this, 'render_field_select' ), $page_slug, 'wpcm_design_section', array( 'id' => 'banner_position', 'options' => array( 'bottom-left'  => __( 'Bottom Left Corner', 'wpcm-agency-hub' ), 'bottom-right' => __( 'Bottom Right Corner', 'wpcm-agency-hub' ), 'footer' => __( 'Footer (Full Width Bar)', 'wpcm-agency-hub' ), 'modal' => __( 'Center Screen (Pop-up)', 'wpcm-agency-hub' ) ) ) );
        add_settings_field( 'bg_color', __( 'Banner Background Color', 'wpcm-agency-hub' ), array( $this, 'render_field_color_picker' ), $page_slug, 'wpcm_design_section', array( 'id' => 'bg_color' ) );
        add_settings_field( 'text_color', __( 'Banner Text Color', 'wpcm-agency-hub' ), array( $this, 'render_field_color_picker' ), $page_slug, 'wpcm_design_section', array( 'id' => 'text_color' ) );
        add_settings_field( 'btn_accept_bg_color', __( 'Accept Button Background', 'wpcm-agency-hub' ), array( $this, 'render_field_color_picker' ), $page_slug, 'wpcm_design_section', array( 'id' => 'btn_accept_bg_color' ) );
        add_settings_field( 'btn_reject_bg_color', __( 'Reject Button Background', 'wpcm-agency-hub' ), array( $this, 'render_field_color_picker' ), $page_slug, 'wpcm_design_section', array( 'id' => 'btn_reject_bg_color' ) );
        add_settings_field( 'btn_reject_text_color', __( 'Reject Button Text', 'wpcm-agency-hub' ), array( $this, 'render_field_color_picker' ), $page_slug, 'wpcm_design_section', array( 'id' => 'btn_reject_text_color' ) );
        add_settings_field( 'btn_reject_border_color', __( 'Reject Button Border', 'wpcm-agency-hub' ), array( $this, 'render_field_color_picker' ), $page_slug, 'wpcm_design_section', array( 'id' => 'btn_reject_border_color' ) );
        
        // --- Seção de Conteúdo ---
        $page_slug_content = 'wpcm_hub_settings_page_content';
        
        // NOVO: Seção Geral dentro de Conteúdo para o GCM
        add_settings_section( 'wpcm_general_content_section', __( 'General Behavior', 'wpcm-agency-hub' ), null, $page_slug_content );
        
        add_settings_field( 'enable_gcm_advanced', __( 'Google Consent Mode v2 (Advanced)', 'wpcm-agency-hub' ), array( $this, 'render_field_checkbox' ), $page_slug_content, 'wpcm_general_content_section', 
            array( 
                'id' => 'enable_gcm_advanced', 
                'label' => __( 'Enable Advanced Mode (Cookieless Pings).', 'wpcm-agency-hub' ),
                'desc' => __( 'If enabled, Google tags will load anonymously before consent. If disabled (default), tags are completely blocked until consent (Basic Mode).', 'wpcm-agency-hub' )
            ) 
        );

        add_settings_section( 'wpcm_banner_content_section', __( 'Banner & Modal Content', 'wpcm-agency-hub' ), null, $page_slug_content );
        add_settings_field( 'policy_page_url', __( 'Privacy Policy URL', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'policy_page_url', 'desc' => __( 'Use {site_url} to automatically insert the client\'s site URL.', 'wpcm-agency-hub' ) ) );
        add_settings_field( 'banner_text', __( 'Banner Main Text', 'wpcm-agency-hub' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'banner_text' ) );
        add_settings_field( 'btn_accept_text', __( 'Accept Button Text', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_accept_text' ) );
        add_settings_field( 'btn_reject_text', __( 'Reject Button Text', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_reject_text' ) );
        add_settings_field( 'btn_settings_text', __( 'Customize Button Text', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_settings_text' ) );
        add_settings_field( 'modal_title', __( 'Modal Title Text', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'modal_title' ) );
        add_settings_field( 'btn_save_text', __( 'Modal Save Button Text', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_save_text' ) );
        
        add_settings_section( 'wpcm_iframe_content_section', __( 'iFrame Blocker Content', 'wpcm-agency-hub' ), null, $page_slug_content );
        add_settings_field( 'enable_iframe_blocker', __( 'iFrame Blocker', 'wpcm-agency-hub' ), array( $this, 'render_field_checkbox' ), $page_slug_content, 'wpcm_iframe_content_section', array( 'id' => 'enable_iframe_blocker', 'label' => __( 'Enable iFrame Blocker (YouTube, Vimeo, Google Maps).', 'wpcm-agency-hub' ) ) );
        add_settings_field( 'iframe_blocker_text', __( 'iFrame Blocker Text', 'wpcm-agency-hub' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_iframe_content_section', array( 'id' => 'iframe_blocker_text' ) );
        add_settings_field( 'iframe_blocker_button_text', __( 'iFrame Blocker Button', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_iframe_content_section', array( 'id' => 'iframe_blocker_button_text' ) );
        add_settings_section( 'wpcm_necessary_content_section', __( 'Necessary Content', 'wpcm-agency-hub' ), null, $page_slug_content );
        add_settings_field( 'necessary_desc', __( 'Necessary Description', 'wpcm-agency-hub' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_necessary_content_section', array( 'id' => 'necessary_desc' ) );

        // --- Seção de Branding & Data ---
        $page_slug_branding = 'wpcm_hub_settings_page_branding';
        add_settings_section( 'wpcm_branding_section', __( 'Agency Branding (Public)', 'wpcm-agency-hub' ), array( $this, 'render_branding_section_text' ), $page_slug_branding );
        add_settings_field( 'agency_branding_name', __( 'Public Agency Name', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_branding, 'wpcm_branding_section', array( 'id' => 'agency_branding_name', 'desc' => __( 'The text displayed on the client\'s banner (e.g., "Powered by My Agency").', 'wpcm-agency-hub' ) ) );
        
        // FIX: Moved URL out of translation string using sprintf
        add_settings_field( 
            'agency_branding_url', 
            __( 'Public Agency URL', 'wpcm-agency-hub' ), 
            array( $this, 'render_field_text' ), 
            $page_slug_branding, 
            'wpcm_branding_section', 
            array( 
                'id' => 'agency_branding_url', 
                'desc' => sprintf( 
                    /* translators: %s: Example URL */
                    __( '(Ex: %s). We recommend using UTM parameters to measure clicks in your Google Analytics.', 'wpcm-agency-hub' ),
                    'https://agency.com?utm_source=plugin'
                ) 
            ) 
        );
        
        add_settings_section( 'wpcm_whitelabel_section', __( 'White-Label (Internal)', 'wpcm-agency-hub' ), array( $this, 'render_whitelabel_section_text' ), $page_slug_branding );
        
        add_settings_field( 'agency_whitelabel_name', __( 'Plugin Name (on client site)', 'wpcm-agency-hub' ), array( $this, 'render_field_text' ), $page_slug_branding, 'wpcm_whitelabel_section', array( 'id' => 'agency_whitelabel_name', 'desc' => __( 'Replaces the default plugin name ("Consent Manager - GDPR & CCPA") in the client\'s plugin list.', 'wpcm-agency-hub' ) ) );
        
        add_settings_field( 'agency_whitelabel_desc', __( 'Plugin Description (on client site)', 'wpcm-agency-hub' ), array( $this, 'render_field_textarea' ), $page_slug_branding, 'wpcm_whitelabel_section', array( 'id' => 'agency_whitelabel_desc', 'rows' => 3 ) );
        add_settings_field( 'agency_whitelabel_hide', __( 'Hide Settings (on client site)', 'wpcm-agency-hub' ), array( $this, 'render_field_checkbox' ), $page_slug_branding, 'wpcm_whitelabel_section', array( 'id' => 'agency_whitelabel_hide', 'label' => __( 'Hide the "Consent Manager" menu from the client. (The plugin will still work).', 'wpcm-agency-hub' ) ) );
        
        add_settings_section( 'wpcm_data_retention_section', __( 'Data & Retention', 'wpcm-agency-hub' ), array( $this, 'render_data_retention_section_text' ), $page_slug_branding );
        add_settings_field( 'hub_log_retention_period', __( 'Global Log Retention', 'wpcm-agency-hub' ), array( $this, 'render_field_select' ), $page_slug_branding, 'wpcm_data_retention_section', array( 'id' => 'hub_log_retention_period', 'desc' => __( 'Automatically delete global consent logs older than this period. "Forever" is not recommended.', 'wpcm-agency-hub' ), 'options' => array( '0'  => __( 'Forever (No Deletion)', 'wpcm-agency-hub' ), '3'  => __( '3 Months', 'wpcm-agency-hub' ), '6'  => __( '6 Months', 'wpcm-agency-hub' ), '12' => __( '12 Months (Recommended)', 'wpcm-agency-hub' ), '24' => __( '24 Months', 'wpcm-agency-hub' ), ) ) );

        add_settings_field( 'enable_ip_logging', __( 'IP Address Logging', 'wpcm-agency-hub' ), array( $this, 'render_field_checkbox' ), $page_slug_branding, 'wpcm_data_retention_section', 
            array( 
                'id' => 'enable_ip_logging', 
                'label' => __( 'Enable IP Address storage in Consent Logs.', 'wpcm-agency-hub' ),
                'desc' => __( 'If enabled, user IP addresses will be stored. Check your local privacy laws (GDPR/LGPD) before enabling.', 'wpcm-agency-hub' )
            ) 
        );
    }
    
    public function render_branding_section_text() { echo '<p>' . esc_html__( 'This branding will be shown on the consent banner of all connected client sites.', 'wpcm-agency-hub' ) . '</p>'; }
    public function render_whitelabel_section_text() { echo '<p>' . esc_html__( 'These settings change how your plugin appears inside the client\'s WordPress dashboard.', 'wpcm-agency-hub' ) . '</p>'; }
    public function render_data_retention_section_text() { echo '<p>' . esc_html__( 'Manage the retention policy for all logs synced to the Hub.', 'wpcm-agency-hub' ) . '</p>'; }

    // Funções de sanitização (Não contém strings)
    private function _get_allowed_script_html() { return array( 'script' => array( 'src' => true, 'type' => true, 'async' => true, 'defer' => true, 'charset' => true, 'id' => true, 'class' => true ) ); }
    private function _sanitize_scripts_field( $input ) { return wp_kses( $input, $this->_get_allowed_script_html() ); }
    public function sanitize_options( $input ) {
        $clean_input = (array) get_option( 'wpcm_hub_global_settings' );
        $defaults = self::get_default_options();
        if ( ! is_array( $clean_input ) ) $clean_input = array();
        
        if ( isset( $input['bg_color'] ) ) {
            $allowed_positions = array( 'footer', 'modal', 'bottom-left', 'bottom-right' );
            $clean_input['banner_position'] = ( isset( $input['banner_position'] ) && in_array( $input['banner_position'], $allowed_positions ) ) ? $input['banner_position'] : $defaults['banner_position'];
            $clean_input['bg_color']        = isset( $input['bg_color'] ) ? sanitize_hex_color( $input['bg_color'] ) : $defaults['bg_color'];
            $clean_input['text_color']      = isset( $input['text_color'] ) ? sanitize_hex_color( $input['text_color'] ) : $defaults['text_color'];
            $clean_input['btn_accept_bg_color']     = isset( $input['btn_accept_bg_color'] ) ? sanitize_hex_color( $input['btn_accept_bg_color'] ) : $defaults['btn_accept_bg_color'];
            $clean_input['btn_reject_bg_color']     = isset( $input['btn_reject_bg_color'] ) ? sanitize_hex_color( $input['btn_reject_bg_color'] ) : $defaults['btn_reject_bg_color'];
            $clean_input['btn_reject_text_color']   = isset( $input['btn_reject_text_color'] ) ? sanitize_hex_color( $input['btn_reject_text_color'] ) : $defaults['btn_reject_text_color'];
            $clean_input['btn_reject_border_color'] = isset( $input['btn_reject_border_color'] ) ? sanitize_hex_color( $input['btn_reject_border_color'] ) : $defaults['btn_reject_border_color'];
        }
        if ( isset( $input['banner_text'] ) ) {
            $clean_input['policy_page_url'] = isset( $input['policy_page_url'] ) ? esc_url_raw( $input['policy_page_url'] ) : $defaults['policy_page_url'];
            $clean_input['banner_text']     = isset( $input['banner_text'] ) ? sanitize_textarea_field( $input['banner_text'] ) : $defaults['banner_text'];
            $clean_input['btn_accept_text'] = isset( $input['btn_accept_text'] ) ? sanitize_text_field( $input['btn_accept_text'] ) : $defaults['btn_accept_text'];
            $clean_input['btn_reject_text'] = isset( $input['btn_reject_text'] ) ? sanitize_text_field( $input['btn_reject_text'] ) : $defaults['btn_reject_text'];
            $clean_input['btn_settings_text'] = isset( $input['btn_settings_text'] ) ? sanitize_text_field( $input['btn_settings_text'] ) : $defaults['btn_settings_text'];
            $clean_input['modal_title']       = isset( $input['modal_title'] ) ? sanitize_text_field( $input['modal_title'] ) : $defaults['modal_title'];
            $clean_input['btn_save_text']     = isset( $input['btn_save_text'] ) ? sanitize_text_field( $input['btn_save_text'] ) : $defaults['btn_save_text'];
            $clean_input['enable_iframe_blocker'] = isset( $input['enable_iframe_blocker'] ) ? '1' : '0';
            $clean_input['enable_gcm_advanced'] = isset( $input['enable_gcm_advanced'] ) ? '1' : '0'; 
            $clean_input['iframe_blocker_text'] = isset( $input['iframe_blocker_text'] ) ? sanitize_textarea_field( $input['iframe_blocker_text'] ) : $defaults['iframe_blocker_text'];
            $clean_input['iframe_blocker_button_text'] = isset( $input['iframe_blocker_button_text'] ) ? sanitize_text_field( $input['iframe_blocker_button_text'] ) : $defaults['iframe_blocker_button_text'];
            $clean_input['necessary_desc']    = isset( $input['necessary_desc'] ) ? sanitize_textarea_field( $input['necessary_desc'] ) : $defaults['necessary_desc'];
        }
        if ( isset( $input['agency_branding_name'] ) || isset( $input['agency_branding_url'] ) || isset( $input['agency_whitelabel_name'] ) || isset($input['hub_log_retention_period']) ) {
            $clean_input['agency_branding_name'] = isset( $input['agency_branding_name'] ) ? sanitize_text_field( $input['agency_branding_name'] ) : '';
            $clean_input['agency_branding_url']  = isset( $input['agency_branding_url'] ) ? esc_url_raw( $input['agency_branding_url'] ) : '';
            $clean_input['agency_whitelabel_name'] = isset( $input['agency_whitelabel_name'] ) ? sanitize_text_field( $input['agency_whitelabel_name'] ) : '';
            $clean_input['agency_whitelabel_desc'] = isset( $input['agency_whitelabel_desc'] ) ? sanitize_textarea_field( $input['agency_whitelabel_desc'] ) : '';
            $clean_input['agency_whitelabel_hide'] = isset( $input['agency_whitelabel_hide'] ) ? '1' : '0';
            $allowed_periods = array( '0', '3', '6', '12', '24' );
            $clean_input['hub_log_retention_period'] = ( isset( $input['hub_log_retention_period'] ) && in_array( $input['hub_log_retention_period'], $allowed_periods ) ) ? $input['hub_log_retention_period'] : '12';
            
            $clean_input['enable_ip_logging'] = isset( $input['enable_ip_logging'] ) ? '1' : '0';
        }
        return $clean_input;
    }

    // Funções de renderização de campos
    public function render_field_text( $args ) { $field_id = $args['id']; $desc = $args['desc'] ?? ''; $value = $this->options[$field_id] ?? ''; printf( '<input type="text" id="%s" name="wpcm_hub_global_settings[%s]" value="%s" class="large-text" />', esc_attr( $field_id ), esc_attr( $field_id ), esc_attr($value) ); if ($desc) { printf('<p class="description">%s</p>', esc_html($desc)); } }
    public function render_field_checkbox( $args ) { $field_id = $args['id']; $label = $args['label'] ?? ''; $desc = $args['desc'] ?? ''; $current_value = $this->options[$field_id] ?? ''; printf( '<fieldset><input type="checkbox" id="%s" name="wpcm_hub_global_settings[%s]" value="1" %s />', esc_attr( $field_id ), esc_attr( $field_id ), checked( $current_value, '1', false ) ); if ($label) { printf('<label for="%s"> %s</label>', esc_attr( $field_id ), esc_html($label)); } if ($desc) { printf('<p class="description">%s</p>', esc_html($desc)); } echo '</fieldset>'; }
    public function render_field_textarea( $args ) { $field_id = $args['id']; $desc = $args['desc'] ?? ''; $rows = $args['rows'] ?? '6'; $value = $this->options[$field_id] ?? ''; printf( '<textarea id="%s" name="wpcm_hub_global_settings[%s]" rows="%d" class="large-text">%s</textarea>', esc_attr( $field_id ), esc_attr( $field_id ), (int) $rows, esc_textarea($value) ); if ($desc) { printf('<p class="description">%s</p>', esc_html($desc)); } }
    public function render_field_select( $args ) { $field_id = $args['id']; $desc = $args['desc'] ?? ''; $options = $args['options'] ?? array(); $selected_value = $this->options[$field_id] ?? ''; printf( '<select id="%s" name="wpcm_hub_global_settings[%s]">', esc_attr( $field_id ), esc_attr( $field_id ) ); foreach ( $options as $value => $label ) { printf( '<option value="%s" %s>%s</option>', esc_attr( $value ), selected( $selected_value, $value, false ), esc_html( $label ) ); } printf( '</select>' ); if ($desc) { printf('<p class="description">%s</p>', esc_html($desc)); } }
    public function render_field_color_picker( $args ) { $field_id = $args['id']; $value = $this->options[$field_id] ?? ''; $default = self::get_default_options()[$field_id] ?? '#000000'; printf( '<input type="text" id="%s" name="wpcm_hub_global_settings[%s]" value="%s" class="wpcm-color-picker" data-default-color="%s" />', esc_attr( $field_id ), esc_attr( $field_id ), esc_attr($value), esc_attr($default) ); }
}