<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Lida com o AGENDAMENTO de trabalhos em segundo plano.
 * Esta classe não executa o trabalho, apenas o coloca na fila.
 */
class WPCM_Hub_Job_Scheduler {

	/**
	 * Agenda uma sincronização para TODOS os sites conectados.
	 *
	 * @return array {'success' => int, 'error' => int}
	 */
	public static function schedule_global_sync() {
		$site_ids = self::get_site_ids_for_query( array() );
		return self::schedule_sync_for_sites( $site_ids );
	}

	/**
	 * Agenda uma sincronização para um único site.
	 *
	 * @param int $site_id
	 * @return array
	 */
	public static function schedule_sync_for_site( $site_id ) {
		return self::schedule_sync_for_sites( array( $site_id ) );
	}

	/**
	 * Agenda uma sincronização para todos os sites que usam um Perfil.
	 *
	 * @param int $profile_id
	 * @return array
	 */
	public static function schedule_sync_for_profile( $profile_id ) {
		$site_ids = self::get_site_ids_for_query(
			array(
				'key'     => '_wpcm_assigned_profile',
				'value'   => $profile_id,
				'compare' => '=',
			)
		);
		return self::schedule_sync_for_sites( $site_ids );
	}

	/**
	 * Função principal que agenda os trabalhos na fila.
	 *
	 * @param array $site_ids Lista de IDs de post de site.
	 * @return array Contagem de sucesso/erro.
	 */
	private static function schedule_sync_for_sites( $site_ids ) {
		if ( empty( $site_ids ) || ! is_array( $site_ids ) ) {
			return array( 'success' => 0, 'error' => 0 );
		}

		$scheduled_count = 0;
		$skipped_count   = 0;

		foreach ( $site_ids as $site_id ) {
			// Verifica se já existe um job pendente para este site
			$pending_jobs = as_get_scheduled_actions(
				array(
					'hook'   => 'wpcm_hub_sync_site_job',
					'args'   => array( 'site_id' => $site_id ),
					'status' => ActionScheduler_Store::STATUS_PENDING,
				)
			);

			// Se não houver job pendente, agenda um novo
			if ( empty( $pending_jobs ) ) {
				as_schedule_single_action(
					time() + 10, // 10 segundos no futuro (para agrupar)
					'wpcm_hub_sync_site_job', // O NOME DO NOSSO JOB (que vamos criar)
					array( 'site_id' => $site_id ),
					'wpcm-hub-sync' // Nosso grupo de jobs
				);
				$scheduled_count++;
			} else {
				$skipped_count++; // Já está na fila, não agenda de novo
			}
		}

		return array( 'success' => $scheduled_count, 'skipped' => $skipped_count );
	}

	/**
	 * Busca os IDs de sites conectados com base em um meta_query.
	 *
	 * @param array $meta_query_item Meta query adicional.
	 * @return array Lista de IDs de post.
	 */
	private static function get_site_ids_for_query( $meta_query_item ) {
		$meta_query = array(
			'relation' => 'AND',
			// Sempre busca apenas sites conectados
			array(
				'key'     => '_status',
				'value'   => 'connected',
				'compare' => '=',
			),
		);

		if ( ! empty( $meta_query_item ) ) {
			$meta_query[] = $meta_query_item;
		}

		$args = array(
			'post_type'              => 'wpcm_hub_site',
			'post_status'            => 'publish',
			'posts_per_page'         => -1,
			'fields'                 => 'ids',
			'no_found_rows'          => true,  // Otimização: Não conta total de linhas
			'update_post_meta_cache' => false, // Otimização: Não carrega meta cache desnecessário
			'update_post_term_cache' => false, // Otimização: Não carrega termos
			'meta_query'             => $meta_query,
		);
		
		// CORREÇÃO: Flag ignorar adicionada exatamente antes da chamada
		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
		return get_posts( $args );
	}
}