<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * ATUALIZADO (Sprint 6 + IP Update): Adiciona Gráficos, endpoint AJAX e Exportação CSV com IP.
 */
class WPCM_Hub_Log_Dashboard {

    private $log_table;

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_dashboard_page' ) );
        add_action( 'admin_init', array( $this, 'handle_csv_export' ) );
        add_action( 'wp_ajax_wpcm_hub_get_chart_data', array( $this, 'ajax_get_chart_data' ) );
    }

    public function add_dashboard_page() {
        $hook_log = add_submenu_page(
            'edit.php?post_type=wpcm_hub_site',
            __( 'Log Dashboard', 'wpcm-agency-hub' ),
            __( 'Log Dashboard', 'wpcm-agency-hub' ),
            'manage_options',
            'wpcm_hub_logs',
            array( $this, 'render_dashboard_page' )
        );
        
        add_action( 'load-' . $hook_log, array( $this, 'instantiate_log_table' ) );
    }
    
    public function handle_csv_export() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'wpcm_hub_logs' || 
             ! isset( $_GET['wpcm_hub_action'] ) || $_GET['wpcm_hub_action'] !== 'export_csv' ) {
            return;
        }

        if ( ! isset( $_GET['wpcm_export_nonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['wpcm_export_nonce'] ), 'wpcm_hub_export_csv_nonce' ) ) {
            wp_die( 'Invalid nonce.' );
        }
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Permission denied.' );
        }

        $filename = 'hub-consent-log-' . gmdate( 'Y-m-d' ) . '.csv';
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $filename );

        $output = fopen( 'php://output', 'w' );
        
        fputcsv( $output, array( 
            'Site Name', 'User (Hashed)', 'IP Address', 'Date/Time (UTC)', 'Policy Version', 'Consented Categories' 
        ) );

        global $wpdb;
        $log_table = $wpdb->prefix . 'wpcm_hub_logs';
        $posts_table = $wpdb->prefix . 'posts';
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $filter_site_id = isset( $_GET['filter_site_id'] ) ? absint( $_GET['filter_site_id'] ) : 0;
        
        $where_clauses = array( '1=1' );
        $prepare_args = array();

        if ( $filter_site_id > 0 ) {
            $where_clauses[] = 't_log.site_post_id = %d';
            $prepare_args[] = $filter_site_id;
        }
        $where_sql = implode( ' AND ', $where_clauses );

        $batch_size = 1000; 
        $page = 1;

        do {
            $offset = ( $page - 1 ) * $batch_size;
            
            $batch_prepare_args = $prepare_args;
            $batch_prepare_args[] = $batch_size; // Limit
            $batch_prepare_args[] = $offset;     // Offset

            // CORREÇÃO: Adicionado ReplacementsWrongNumber para silenciar o aviso de contagem de arrays.
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
            $items = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT t_log.*, t_posts.post_title AS site_name 
                     FROM $log_table AS t_log
                     LEFT JOIN $posts_table AS t_posts ON t_log.site_post_id = t_posts.ID
                     WHERE $where_sql
                     ORDER BY t_log.timestamp DESC
                     LIMIT %d OFFSET %d",
                    $batch_prepare_args
                )
            );

            if ( ! empty( $items ) ) {
                foreach ( $items as $row ) {
                    fputcsv( $output, array(
                        $row->site_name, $row->user_hash, ! empty( $row->ip_address ) ? $row->ip_address : '', 
                        $row->timestamp, $row->policy_version, $row->consent_categories
                    ) );
                }
                $page++;
            }

        } while ( ! empty( $items ) && count( $items ) === $batch_size ); 
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
        fclose( $output );
        exit;
    }


    public function instantiate_log_table() {
        if ( ! class_exists( 'WPCM_Hub_Log_Table' ) ) {
            // O Autoloader cuida disso
        }
        $this->log_table = new WPCM_Hub_Log_Table();
    }

    private function render_stats_dashboard() {
        global $wpdb;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $filter_site_id = isset( $_GET['filter_site_id'] ) ? absint( $_GET['filter_site_id'] ) : 0;
        $cache_key = 'wpcm_hub_stats_' . $filter_site_id;
        $stats = wp_cache_get( $cache_key, 'wpcm_hub_consent' );

        if ( false === $stats ) {
            $log_table = $wpdb->prefix . 'wpcm_hub_logs';
            $where_clauses = array( '1=1' );
            
            $prepare_args = array( '%analytics%', '%marketing%' );
            
            if ( $filter_site_id > 0 ) {
                $where_clauses[] = "site_post_id = %d";
                $prepare_args[] = $filter_site_id;
            }
            $where_sql = implode( ' AND ', $where_clauses );
            
            // CORREÇÃO: Adicionado ReplacementsWrongNumber para cobrir a contagem de arrays.
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
            $stats = $wpdb->get_row( 
                $wpdb->prepare( 
                    "SELECT 
                        COUNT(id) as total, 
                        SUM(CASE WHEN consent_categories LIKE %s AND consent_categories LIKE %s THEN 1 ELSE 0 END) as full_consent, 
                        SUM(CASE WHEN consent_categories = 'necessary' THEN 1 ELSE 0 END) as necessary_only 
                    FROM $log_table
                    WHERE $where_sql", 
                    $prepare_args 
                ), 
                ARRAY_A 
            );
            wp_cache_set( $cache_key, $stats, 'wpcm_hub_consent', 1 * HOUR_IN_SECONDS );
        }
        
        $total = (int) ( $stats['total'] ?? 0 ); 
        $full = (int) ( $stats['full_consent'] ?? 0 ); 
        $necessary = (int) ( $stats['necessary_only'] ?? 0 ); 
        $partial = max( 0, $total - $full - $necessary ); 
        $full_rate = $total > 0 ? round( ($full / $total) * 100 ) : 0;
        
        ?>
        <style>
            .wpcm-dashboard-stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 20px; }
            .wpcm-stat-card { background: #fff; border: 1px solid #e0e0e0; border-left-width: 4px; padding: 16px; display: flex; align-items: center; }
            .wpcm-stat-icon { font-size: 24px; color: #777; margin-right: 16px; }
            .wpcm-stat-content { display: flex; flex-direction: column; }
            .wpcm-stat-number { font-size: 24px; font-weight: 600; line-height: 1.2; }
            .wpcm-stat-label { font-size: 14px; color: #555; }
            .wpcm-border-blue { border-left-color: #3498db; }
            .wpcm-border-green { border-left-color: #2ecc71; }
            .wpcm-border-orange { border-left-color: #f39c12; }
            .wpcm-border-red { border-left-color: #e74c3c; }
        </style>

        <div class="wpcm-dashboard-stats">
            <div class="wpcm-stat-card wpcm-border-blue"><div class="wpcm-stat-icon dashicons dashicons-chart-area"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $total ) ); ?></span><span class="wpcm-stat-label"><?php esc_html_e( 'Total Interactions', 'wpcm-agency-hub' ); ?></span></div></div>
            <div class="wpcm-stat-card wpcm-border-green"><div class="wpcm-stat-icon dashicons dashicons-yes-alt"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $full ) ); ?> <small>(<?php echo esc_html( $full_rate ); ?>%)</small></span><span class="wpcm-stat-label"><?php esc_html_e( 'Accepted All', 'wpcm-agency-hub' ); ?></span></div></div>
            <div class="wpcm-stat-card wpcm-border-orange"><div class="wpcm-stat-icon dashicons dashicons-filter"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $partial ) ); ?></span><span class="wpcm-stat-label"><?php esc_html_e( 'Partially Accepted', 'wpcm-agency-hub' ); ?></span></div></div>
            <div class="wpcm-stat-card wpcm-border-red"><div class="wpcm-stat-icon dashicons dashicons-dismiss"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $necessary ) ); ?></span><span class="wpcm-stat-label"><?php esc_html_e( 'Rejected (Necessary)', 'wpcm-agency-hub' ); ?></span></div></div>
        </div>

        <div class="wpcm-charts-wrapper">
            <div class="wpcm-chart-container">
                <h3><?php esc_html_e( 'Consent Breakdown', 'wpcm-agency-hub' ); ?></h3>
                <div id="wpcm-pie-chart-loading" class="wpcm-chart-loading-spinner">
                    <span class="spinner"></span>
                    <?php esc_html_e( 'Loading Chart...', 'wpcm-agency-hub' ); ?>
                </div>
                <div class="wpcm-chart-inner">
                    <canvas id="wpcm-pie-chart-canvas"></canvas>
                </div>
            </div>
            <div class="wpcm-chart-container">
                <h3><?php esc_html_e( 'Interactions (Last 14 Days)', 'wpcm-agency-hub' ); ?></h3>
                <div id="wpcm-line-chart-loading" class="wpcm-chart-loading-spinner">
                    <span class="spinner"></span>
                    <?php esc_html_e( 'Loading Chart...', 'wpcm-agency-hub' ); ?>
                </div>
                <div class="wpcm-chart-inner">
                    <canvas id="wpcm-line-chart-canvas"></canvas>
                </div>
            </div>
        </div>
        <?php
    }

    public function render_dashboard_page() {
        $this->log_table->prepare_items();
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e( 'Global Consent Log', 'wpcm-agency-hub' ); ?></h1>
            <p><?php esc_html_e( 'View consent logs collected from all connected client sites.', 'wpcm-agency-hub' ); ?></p>
            
            <?php $this->render_stats_dashboard(); ?>
            
            <hr class="wp-header-end">
            <form id="consent-log-form" method="get">
                <input type="hidden" name="post_type" value="wpcm_hub_site" />
                <input type="hidden" name="page" value="wpcm_hub_logs" />
                
                <?php
                $this->log_table->display();
                ?>
            </form>
        </div>
        <?php
    }

    public function ajax_get_chart_data() {
        check_ajax_referer( 'wpcm_hub_chart_nonce', '_ajax_nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Permission denied.' ) );
        }

        global $wpdb;
        $log_table = $wpdb->prefix . 'wpcm_hub_logs';
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $filter_site_id = isset( $_GET['filter_site_id'] ) ? absint( $_GET['filter_site_id'] ) : 0;
        
        $pie_prepare_args = array( '%analytics%', '%marketing%', '%analytics%', '%marketing%' );
        $where_clauses = array( '1=1' );
        
        if ( $filter_site_id > 0 ) {
            $where_clauses[] = "site_post_id = %d";
            $pie_prepare_args[] = $filter_site_id;
        }
        $where_sql = implode( ' AND ', $where_clauses );

        // CORREÇÃO: Adicionado ReplacementsWrongNumber para cobrir a contagem de arrays.
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        $pie_data = $wpdb->get_row( 
            $wpdb->prepare( 
                "SELECT 
                    SUM(CASE WHEN consent_categories LIKE %s AND consent_categories LIKE %s THEN 1 ELSE 0 END) as full_consent, 
                    SUM(CASE WHEN consent_categories = 'necessary' THEN 1 ELSE 0 END) as necessary_only,
                    (COUNT(id) - SUM(CASE WHEN consent_categories LIKE %s AND consent_categories LIKE %s THEN 1 ELSE 0 END) - SUM(CASE WHEN consent_categories = 'necessary' THEN 1 ELSE 0 END)) as partial
                FROM $log_table
                WHERE $where_sql",
                $pie_prepare_args
            ), 
            ARRAY_A 
        );

        $pie_chart = array(
            'labels' => array(
                __( 'Accepted All', 'wpcm-agency-hub' ),
                __( 'Rejected (Necessary)', 'wpcm-agency-hub' ),
                __( 'Partially Accepted', 'wpcm-agency-hub' ),
            ),
            'data'   => array(
                (int) ( $pie_data['full_consent'] ?? 0 ),
                (int) ( $pie_data['necessary_only'] ?? 0 ),
                (int) ( $pie_data['partial'] ?? 0 ),
            ),
        );

        $line_prepare_args = array();
        $line_prepare_args[] = gmdate( 'Y-m-d 00:00:00', strtotime( '-13 days' ) ); 
        
        if ( $filter_site_id > 0 ) {
            $line_prepare_args[] = $filter_site_id;
        }

        $log_table_name = $wpdb->prefix . 'wpcm_hub_logs';
        $sql_line_base = "SELECT DATE(timestamp) as date, COUNT(id) as total FROM $log_table_name WHERE timestamp >= %s";
        
        if ( $filter_site_id > 0 ) {
            $sql_line_base .= " AND site_post_id = %d";
        }
        
        $sql_line_base .= " GROUP BY DATE(timestamp) ORDER BY date ASC";

        // CORREÇÃO: Adicionado NotPrepared e ReplacementsWrongNumber
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
        $line_results = $wpdb->get_results( 
            $wpdb->prepare( 
                $sql_line_base,
                $line_prepare_args
            ) 
        );

        $line_data_map = array();
        foreach ( $line_results as $row ) {
            $line_data_map[ $row->date ] = (int) $row->total;
        }

        $line_labels = array();
        $line_data = array();
        for ( $i = 13; $i >= 0; $i-- ) {
            $date_key = gmdate( 'Y-m-d', strtotime( "-$i days" ) );
            $line_labels[] = gmdate( 'M j', strtotime( "-$i days" ) ); 
            $line_data[] = $line_data_map[ $date_key ] ?? 0;
        }

        $line_chart = array(
            'labels' => $line_labels,
            'data'   => $line_data,
        );

        wp_send_json_success(
            array(
                'pie_chart'  => $pie_chart,
                'line_chart' => $line_chart,
            )
        );
    }
}