<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gerencia os Endpoints da REST API para o Agency Hub.
 * (Refatorado para Inserção em Massa de Logs + IP Logging + Limpeza de Buffer)
 */
class WPCM_Hub_API {

    protected $namespace = 'wpcm-hub/v1';

    public function __construct() {
        add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
    }

    public function register_rest_routes() {
        
        register_rest_route( $this->namespace, '/connect', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'handle_connection_attempt' ),
            'permission_callback' => '__return_true',
        ) );

        register_rest_route( $this->namespace, '/settings', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'handle_get_settings' ),
            'permission_callback' => array( $this, 'permission_check' ),
        ) );

        register_rest_route( $this->namespace, '/sync_logs', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'handle_sync_logs' ),
            'permission_callback' => array( $this, 'permission_check' ),
        ) );
    }

    public function handle_connection_attempt( $request ) {
        // Limpa qualquer lixo (PHP Notices) que tenha sido impresso antes
        if ( ob_get_length() ) {
            ob_clean();
        }

        $api_key = sanitize_text_field( $request->get_param( 'api_key' ) );
        $site_url = esc_url_raw( $request->get_param( 'site_url' ) );

        if ( empty( $api_key ) || empty( $site_url ) ) {
            return new WP_REST_Response( array( 'connected' => false, 'error' => __( 'Missing API Key or Site URL.', 'wpcm-agency-hub' ) ), 400 );
        }
        $site_post = $this->get_site_by_api_key( $api_key );
        if ( ! $site_post ) {
            return new WP_REST_Response( array( 'connected' => false, 'error' => __( 'Invalid API Key.', 'wpcm-agency-hub' ) ), 403 );
        }
        $site_post_id = $site_post->ID;
        $site_title = $site_post->post_title;
        
        update_post_meta( $site_post_id, '_site_url', $site_url );
        update_post_meta( $site_post_id, '_status', 'connected' );

        return new WP_REST_Response( array( 'connected' => true, 'site_name' => $site_title ), 200 );
    }

    public function permission_check( $request ) {
        $api_key = $request->get_header( 'X-Api-Key' );
        if ( empty( $api_key ) ) {
            $api_key = $request->get_param( 'api_key' );
        }
        
        // Sanitiza a chave antes de usar
        $api_key = sanitize_text_field( $api_key );
        
        if ( empty( $api_key ) ) {
            return new WP_Error( 'rest_forbidden', __( 'Missing API Key.', 'wpcm-agency-hub' ), array( 'status' => 401 ) );
        }

        $site_post = $this->get_site_by_api_key( $api_key );
        if ( ! $site_post ) {
            return new WP_Error( 'rest_forbidden', __( 'Invalid API Key.', 'wpcm-agency-hub' ), array( 'status' => 403 ) );
        }
        
        $request->set_param( 'site_post_id', $site_post->ID );
        return true;
    }

    public function handle_get_settings( $request ) {
        
        if ( ob_get_length() ) ob_clean();

        $global_settings = get_option( 'wpcm_hub_global_settings', array() );
        $defaults = WPCM_Hub_Settings::get_default_options();
        $global_settings = wp_parse_args( $global_settings, $defaults );

        $site_post_id = $request->get_param( 'site_post_id' );
        $assigned_profile_id = (int) get_post_meta( $site_post_id, '_wpcm_assigned_profile', true );

        $settings_source = $global_settings;
        $service_ids_to_load = array();

        if ( $assigned_profile_id > 0 && get_post_status( $assigned_profile_id ) === 'publish' ) {
            $profile_data = get_post_meta( $assigned_profile_id, '_wpcm_profile_settings', true );
            if ( ! is_array( $profile_data ) ) $profile_data = array();
            
            $settings_source = wp_parse_args( $profile_data, $defaults );
            $service_ids_to_load = $settings_source['services'] ?? array();

        } else {
            $service_ids_to_load = get_posts( array(
                'post_type'      => 'wpcm_hub_service',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'fields'         => 'ids',
            ) );
        }

        $services = array();
        if ( ! empty( $service_ids_to_load ) ) {
            $safe_ids = array_map( 'absint', $service_ids_to_load );
            
            $query = new WP_Query( array(
                'post_type'      => 'wpcm_hub_service',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
                'post__in'       => $safe_ids, 
            ) );

            if ( $query->have_posts() ) {
                while ( $query->have_posts() ) {
                    $query->the_post();
                    $id = get_the_ID();
                    $keywords_str = get_post_meta( $id, '_wpcm_keywords', true );
                    
                    $services[] = array(
                        'id'          => 'service_hub_' . $id,
                        'name'        => get_the_title(),
                        'category'    => get_post_meta( $id, '_wpcm_category', true ),
                        'desc'        => get_post_meta( $id, '_wpcm_description', true ),
                        'keywords'    => array_values( array_filter( array_map( 'trim', explode( "\n", $keywords_str ) ) ) ),
                        'scripts'     => get_post_meta( $id, '_wpcm_scripts', true ),
                    );
                }
                wp_reset_postdata();
            }
        }
        
        $branding = array(
            'agency_name' => $global_settings['agency_branding_name'] ?? '',
            'agency_url'  => $global_settings['agency_branding_url'] ?? '',
        );
        $whitelabel = array(
            'name' => $global_settings['agency_whitelabel_name'] ?? '',
            'desc' => $global_settings['agency_whitelabel_desc'] ?? '',
            'hide' => $global_settings['agency_whitelabel_hide'] ?? '0',
        );

        $response_data = array(
            'settings'   => $settings_source,
            'services'   => $services,
            'branding'   => $branding,
            'whitelabel' => $whitelabel,
        );

        return new WP_REST_Response( $response_data, 200 );
    }

    public function handle_sync_logs( $request ) {
        global $wpdb;
        $site_post_id = absint( $request->get_param( 'site_post_id' ) );
        $logs         = $request->get_json_params();

        if ( empty( $logs ) || ! is_array( $logs ) ) {
            return new WP_REST_Response( array( 'success' => false, 'error' => 'No logs provided.' ), 400 );
        }

        $table_name = $wpdb->prefix . 'wpcm_hub_logs';
        
        // Inicia a construção da query de Bulk Insert
        $base_query = "INSERT IGNORE INTO $table_name (site_post_id, client_log_id, user_hash, timestamp, consent_categories, policy_version, ip_address) VALUES ";

        $placeholders = array();
        $values       = array();
        $total_logs   = 0;

        foreach ( $logs as $log ) {
            if ( empty( $log['id'] ) || empty( $log['user_hash'] ) || empty( $log['timestamp'] ) ) {
                continue;
            }

            array_push(
                $values,
                $site_post_id,
                absint( $log['id'] ),
                sanitize_text_field( $log['user_hash'] ),
                sanitize_text_field( $log['timestamp'] ), 
                sanitize_text_field( $log['consent_categories'] ),
                sanitize_text_field( $log['policy_version'] ),
                sanitize_text_field( $log['ip_address'] ?? '' )
            );
            
            // Adiciona o placeholder para esta linha
            $placeholders[] = '(%d, %d, %s, %s, %s, %s, %s)';
            $total_logs++;
        }

        if ( $total_logs === 0 ) {
             return new WP_REST_Response( array( 'success' => false, 'error' => 'All logs provided were malformed.' ), 400 );
        }

        // Concatena os placeholders
        $query = $base_query . implode( ', ', $placeholders );
        
        // Prepara a query com segurança usando $wpdb->prepare
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- A query é construída dinamicamente para bulk insert, mas todos os valores passam por prepare.
        $safe_query = $wpdb->prepare( $query, $values );
        
        // Executa a query
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $result = $wpdb->query( $safe_query );
        
        $inserted_count = ( $result === false ) ? 0 : $result;

        return new WP_REST_Response( array(
            'success'  => true,
            'inserted' => $inserted_count, 
            'received' => $total_logs,     
        ), 200 );
    }

    private function get_site_by_api_key( $api_key ) {
        // Otimização de performance para evitar Slow Query em meta_query
        $args = array(
            'post_type'              => 'wpcm_hub_site',
            'post_status'            => 'publish',
            'posts_per_page'         => 1,
            'no_found_rows'          => true,  // Não calcula paginação (Performance)
            'update_post_meta_cache' => false, // Não carrega meta dados extras (Performance)
            'update_post_term_cache' => false, // Não carrega termos (Performance)
            'meta_query'             => array(
                array(
                    'key'     => '_api_key',
                    'value'   => sanitize_text_field( $api_key ),
                    'compare' => '=',
                ),
            ),
        );
        
        // CORREÇÃO: Flag ignore aplicada diretamente antes da chamada da WP_Query
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
        $site_query = new WP_Query( $args );

        if ( ! $site_query->have_posts() ) {
            return false;
        }
        return $site_query->posts[0];
    }
}