<?php
/**
 * Settings API Class - Versão Final (Protected Status + Descriptions Restored)
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPCM_Settings_API {

    private $options;

    public function __construct() {
        $saved_options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $saved_options ) ) { $saved_options = array(); }
        $this->options = wp_parse_args( $saved_options, $this->get_default_options() );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
    }

    public static function get_default_options() {
        return array(
            'enable_banner'   => '1',
            'policy_page_url' => '',
            'policy_version'  => '1.0',
            'log_retention_period' => '12',
            'enable_iframe_blocker' => '1',
            'enable_ip_logging' => '0',
            'enable_gcm_advanced' => '0', 
            'agency_connection_status' => 'disconnected',
            'agency_hub_url' => '',
            'agency_api_key' => '',
            'banner_position' => 'bottom-left',
            'bg_color'        => '#FFFFFF',
            'text_color'      => '#404040',
            'btn_accept_bg_color'    => '#4F46E5',
            'btn_reject_bg_color'    => 'transparent',
            'btn_reject_text_color'  => '#404040',
            'btn_reject_border_color' => '#CCCCCC',
            'banner_text'     => __( 'We use cookies on our website to give you the most relevant experience.', 'consent-manager-gdpr-ccpa' ),
            'btn_accept_text' => __( 'Accept All', 'consent-manager-gdpr-ccpa' ),
            'btn_reject_text' => __( 'Reject All', 'consent-manager-gdpr-ccpa' ),
            'btn_settings_text' => __( 'Customize', 'consent-manager-gdpr-ccpa' ),
            'modal_title'       => __( 'Manage Your Consent', 'consent-manager-gdpr-ccpa' ),
            'btn_save_text'     => __( 'Save Preferences', 'consent-manager-gdpr-ccpa' ),
            'iframe_blocker_text'   => __( 'This content is blocked. To view it, you must accept Marketing cookies.', 'consent-manager-gdpr-ccpa' ),
            'iframe_blocker_button_text' => __( 'Accept & View Content', 'consent-manager-gdpr-ccpa' ),
            'necessary_desc'    => __( 'These cookies are essential for the website to function properly.', 'consent-manager-gdpr-ccpa' ),
            'necessary_scripts' => '',
        );
    }

    public function register_settings() {
        register_setting( 'wpcm_settings_group', 'wpcm_settings_options', array( $this, 'sanitize_options' ) );

        $page_slug = 'wpcm_settings_page_general';
        add_settings_section( 'wpcm_general_section', __( 'General Settings', 'consent-manager-gdpr-ccpa' ), null, $page_slug );
        add_settings_field( 'enable_banner', __( 'Enable Consent Banner', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_checkbox' ), $page_slug, 'wpcm_general_section', array( 'id' => 'enable_banner', 'label' => __( 'Activate the banner on the front-end of the site.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'policy_page_url', __( 'Privacy Policy URL', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug, 'wpcm_general_section', array( 'id' => 'policy_page_url' ) );
        add_settings_field( 'policy_version', __( 'Policy Version', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug, 'wpcm_general_section', array( 'id' => 'policy_version', 'desc' => __( 'Change this (e.g., to 1.1) to ask for consent again from all users.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'log_retention_period', __( 'Log Retention', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_select' ), $page_slug, 'wpcm_general_section', array( 'id' => 'log_retention_period', 'desc' => __( 'Automatically delete consent logs older than this period.', 'consent-manager-gdpr-ccpa' ), 'options' => array('0'=>'Forever','3'=>'3 Months','6'=>'6 Months','12'=>'12 Months','24'=>'24 Months') ) );
        add_settings_field( 'enable_gcm_advanced', __( 'Google Consent Mode v2 (Advanced)', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_checkbox' ), $page_slug, 'wpcm_general_section', array( 'id' => 'enable_gcm_advanced', 'label' => __( 'Enable Advanced Mode.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'enable_iframe_blocker', __( 'iFrame Blocker', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_checkbox' ), $page_slug, 'wpcm_general_section', array( 'id' => 'enable_iframe_blocker', 'label' => __( 'Enable iFrame Blocker.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'enable_ip_logging', __( 'Store IP Address', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_checkbox' ), $page_slug, 'wpcm_general_section', array( 'id' => 'enable_ip_logging', 'label' => __( 'Store the user\'s IP address.', 'consent-manager-gdpr-ccpa' ) ) );

        $page_slug_design = 'wpcm_settings_page_design';
        add_settings_section( 'wpcm_design_section', __( 'Banner Design', 'consent-manager-gdpr-ccpa' ), null, $page_slug_design );
        add_settings_field( 'banner_position', __( 'Banner Position', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_select' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'banner_position', 'options' => array( 'bottom-left' => 'Bottom Left', 'bottom-right' => 'Bottom Right', 'footer' => 'Footer', 'modal' => 'Center' ) ) );
        add_settings_field( 'bg_color', __( 'Banner Background Color', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_color_picker' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'bg_color' ) );
        add_settings_field( 'text_color', __( 'Banner Text Color', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_color_picker' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'text_color' ) );
        add_settings_field( 'btn_accept_bg_color', __( 'Accept Button Background', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_color_picker' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'btn_accept_bg_color' ) );
        add_settings_field( 'btn_reject_bg_color', __( 'Reject Button Background', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_color_picker' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'btn_reject_bg_color' ) );
        add_settings_field( 'btn_reject_text_color', __( 'Reject Button Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_color_picker' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'btn_reject_text_color' ) );
        add_settings_field( 'btn_reject_border_color', __( 'Reject Button Border', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_color_picker' ), $page_slug_design, 'wpcm_design_section', array( 'id' => 'btn_reject_border_color' ) );
        
        $page_slug_content = 'wpcm_settings_page_content';
        add_settings_section( 'wpcm_banner_content_section', __( 'Banner & Modal Content', 'consent-manager-gdpr-ccpa' ), null, $page_slug_content );
        add_settings_field( 'banner_text', __( 'Banner Main Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'banner_text' ) );
        add_settings_field( 'btn_accept_text', __( 'Accept Button Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_accept_text' ) );
        add_settings_field( 'btn_reject_text', __( 'Reject Button Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_reject_text' ) );
        add_settings_field( 'btn_settings_text', __( 'Customize Button Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_settings_text' ) );
        add_settings_field( 'modal_title', __( 'Modal Title Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'modal_title' ) );
        add_settings_field( 'btn_save_text', __( 'Modal Save Button Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_banner_content_section', array( 'id' => 'btn_save_text' ) );
        add_settings_section( 'wpcm_iframe_content_section', __( 'iFrame Blocker Content', 'consent-manager-gdpr-ccpa' ), null, $page_slug_content );
        add_settings_field( 'iframe_blocker_text', __( 'iFrame Blocker Text', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_iframe_content_section', array( 'id' => 'iframe_blocker_text', 'desc' => __( 'Message shown on blocked content (e.g., YouTube videos).', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'iframe_blocker_button_text', __( 'iFrame Blocker Button', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_content, 'wpcm_iframe_content_section', array( 'id' => 'iframe_blocker_button_text', 'desc' => __( 'Text for the button that unblocks the content.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_section( 'wpcm_necessary_content_section', __( 'Necessary Content', 'consent-manager-gdpr-ccpa' ), null, $page_slug_content );
        add_settings_field( 'necessary_desc', __( 'Necessary Description', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_necessary_content_section', array( 'id' => 'necessary_desc', 'desc' => __( 'Description for the "Necessary" category, which is always on.', 'consent-manager-gdpr-ccpa') ) );
        add_settings_field( 'necessary_scripts', __( 'Necessary Manual Scripts', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_textarea' ), $page_slug_content, 'wpcm_necessary_content_section', array( 'id' => 'necessary_scripts', 'desc' => __( 'These scripts will always load, regardless of user consent.', 'consent-manager-gdpr-ccpa' ) ) );

        $page_slug_agency = 'wpcm_settings_page_agency';
        add_settings_section( 'wpcm_agency_section', __( 'Agency Hub Connection', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_agency_section_text' ), $page_slug_agency );
        add_settings_field( 'agency_hub_url', __( 'Hub URL', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_agency, 'wpcm_agency_section', array( 'id' => 'agency_hub_url', 'desc' => __( 'The login URL of your central Agency Hub dashboard.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'agency_api_key', __( 'Site API Key', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_text' ), $page_slug_agency, 'wpcm_agency_section', array( 'id' => 'agency_api_key', 'desc' => __( 'Generate this key in your Agency Hub and paste it here.', 'consent-manager-gdpr-ccpa' ) ) );
        add_settings_field( 'agency_connection_status', __( 'Connection Status', 'consent-manager-gdpr-ccpa' ), array( $this, 'render_field_connection_status' ), $page_slug_agency, 'wpcm_agency_section', array( 'id' => 'agency_connection_status' ) );
    }
    
    public function render_category_section_text() { echo '<p>' . esc_html__( 'Configure this consent category.', 'consent-manager-gdpr-ccpa' ) . '</p>'; }
    private function _get_allowed_script_html() { return array( 'script' => array( 'src' => true, 'type' => true, 'async' => true, 'defer' => true, 'charset' => true, 'id' => true, 'class' => true ) ); }
    private function _sanitize_scripts_field( $input ) { return wp_kses( $input, $this->_get_allowed_script_html() ); }

    public function sanitize_options( $input ) {
        wp_cache_delete( 'wpcm_settings_options', 'options' );
        
        global $wpdb;
        $row = $wpdb->get_row( $wpdb->prepare( "SELECT option_value FROM $wpdb->options WHERE option_name = %s", 'wpcm_settings_options' ) );
        $db_options = ( is_object($row) && isset($row->option_value) ) ? maybe_unserialize( $row->option_value ) : array();
        if (!is_array($db_options)) $db_options = array();
        
        $status_from_db = $db_options['agency_connection_status'] ?? 'disconnected';
        $real_url = $db_options['agency_hub_url'] ?? '';
        $real_key = $db_options['agency_api_key'] ?? '';

        // Lógica de Proteção de Status
        $input['agency_connection_status'] = $status_from_db;

        $clean_input = $db_options;
        $new_input = array_merge( $clean_input, $input );

        if ( $status_from_db === 'connected' ) {
             $new_input['agency_connection_status'] = 'connected';
             if ( !empty($real_url) && empty($input['agency_hub_url']) ) $new_input['agency_hub_url'] = $real_url;
             if ( !empty($real_key) && empty($input['agency_api_key']) ) $new_input['agency_api_key'] = $real_key;
        }

        $new_input['enable_banner']   = isset( $new_input['enable_banner'] ) ? '1' : '0';
        $new_input['policy_page_url'] = esc_url_raw( $new_input['policy_page_url'] ?? '' );
        $new_input['policy_version']  = sanitize_text_field( $new_input['policy_version'] ?? '1.0' );
        $new_input['log_retention_period'] = sanitize_text_field( $new_input['log_retention_period'] ?? '12' );
        $new_input['enable_iframe_blocker'] = isset( $new_input['enable_iframe_blocker'] ) ? '1' : '0';
        $new_input['enable_ip_logging'] = isset( $new_input['enable_ip_logging'] ) ? '1' : '0';
        $new_input['enable_gcm_advanced'] = isset( $new_input['enable_gcm_advanced'] ) ? '1' : '0'; 
        
        // Campos de Design e Texto
        $allowed_positions = array( 'footer', 'modal', 'bottom-left', 'bottom-right' );
        $new_input['banner_position'] = in_array( ($new_input['banner_position'] ?? ''), $allowed_positions ) ? $new_input['banner_position'] : self::get_default_options()['banner_position'];
        $new_input['bg_color'] = sanitize_hex_color( $new_input['bg_color'] ?? '#FFFFFF' );
        // ... (sanitização de cores continua a mesma, omitida por brevidade mas deve estar no arquivo final)
        $new_input['text_color']      = sanitize_hex_color( $new_input['text_color'] ?? '#404040' );
        $new_input['btn_accept_bg_color']     = sanitize_hex_color( $new_input['btn_accept_bg_color'] ?? '#4F46E5' );
        $new_input['btn_reject_bg_color']     = sanitize_hex_color( $new_input['btn_reject_bg_color'] ?? 'transparent' );
        $new_input['btn_reject_text_color']   = sanitize_hex_color( $new_input['btn_reject_text_color'] ?? '#404040' );
        $new_input['btn_reject_border_color'] = sanitize_hex_color( $new_input['btn_reject_border_color'] ?? '#CCCCCC' );
        
        $new_input['banner_text']     = sanitize_textarea_field( $new_input['banner_text'] ?? '' );
        $new_input['btn_accept_text'] = sanitize_text_field( $new_input['btn_accept_text'] ?? '' );
        $new_input['btn_reject_text'] = sanitize_text_field( $new_input['btn_reject_text'] ?? '' );
        $new_input['btn_settings_text'] = sanitize_text_field( $new_input['btn_settings_text'] ?? '' );
        $new_input['modal_title']       = sanitize_text_field( $new_input['modal_title'] ?? '' );
        $new_input['btn_save_text']     = sanitize_text_field( $new_input['btn_save_text'] ?? '' );
        $new_input['iframe_blocker_text'] = sanitize_textarea_field( $new_input['iframe_blocker_text'] ?? '' );
        $new_input['iframe_blocker_button_text'] = sanitize_text_field( $new_input['iframe_blocker_button_text'] ?? '' );
        $new_input['necessary_desc']    = sanitize_textarea_field( $new_input['necessary_desc'] ?? '' );
        $new_input['necessary_scripts'] = $this->_sanitize_scripts_field( wp_unslash( $new_input['necessary_scripts'] ?? '' ) );
        
        if ( isset($input['agency_hub_url']) ) $new_input['agency_hub_url'] = esc_url_raw( $input['agency_hub_url'] );
        if ( isset($input['agency_api_key']) ) $new_input['agency_api_key'] = sanitize_text_field( $input['agency_api_key'] );

        if ( class_exists( 'WPCM_Core' ) ) {
            WPCM_Core::clear_frontend_cache();
        }
        
        return $new_input;
    }

    // FUNÇÕES DE RENDERIZAÇÃO RESTAURADAS (Com Descrições!)
    
    public function render_field_text( $args ) { 
        $val = $this->options[$args['id']] ?? ''; 
        printf( '<input type="text" id="%s" name="wpcm_settings_options[%s]" value="%s" class="regular-text" />', esc_attr( $args['id'] ), esc_attr( $args['id'] ), esc_attr($val) ); 
        if ( ! empty( $args['desc'] ) ) { printf( '<p class="description">%s</p>', esc_html( $args['desc'] ) ); }
    }
    
    public function render_field_checkbox( $args ) { 
        $val = $this->options[$args['id']] ?? ''; 
        printf( '<fieldset><input type="checkbox" id="%s" name="wpcm_settings_options[%s]" value="1" %s /> <label for="%s">%s</label>', esc_attr( $args['id'] ), esc_attr( $args['id'] ), checked( $val, '1', false ), esc_attr( $args['id'] ), esc_html( $args['label'] ?? '' ) ); 
        if ( ! empty( $args['desc'] ) ) { printf( '<p class="description">%s</p>', esc_html( $args['desc'] ) ); }
        echo '</fieldset>'; 
    }
    
    public function render_field_textarea( $args ) { 
        $val = $this->options[$args['id']] ?? ''; 
        printf( '<textarea id="%s" name="wpcm_settings_options[%s]" rows="6" class="large-text">%s</textarea>', esc_attr( $args['id'] ), esc_attr( $args['id'] ), esc_textarea($val) ); 
        if ( ! empty( $args['desc'] ) ) { printf( '<p class="description">%s</p>', esc_html( $args['desc'] ) ); }
    }
    
    public function render_field_select( $args ) { 
        $val = $this->options[$args['id']] ?? ''; 
        echo '<select name="wpcm_settings_options['.esc_attr($args['id']).']">'; 
        foreach($args['options'] as $k=>$v){ echo '<option value="'.esc_attr($k).'" '.selected($val,$k,false).'>'.esc_html($v).'</option>'; } 
        echo '</select>'; 
        if ( ! empty( $args['desc'] ) ) { printf( '<p class="description">%s</p>', esc_html( $args['desc'] ) ); }
    }
    
    public function render_field_color_picker( $args ) { 
        $val = $this->options[$args['id']] ?? ''; 
        $default = self::get_default_options()[$args['id']] ?? '#000000'; 
        printf( '<input type="text" id="%s" name="wpcm_settings_options[%s]" value="%s" class="wpcm-color-picker" data-default-color="%s" />', esc_attr( $args['id'] ), esc_attr( $args['id'] ), esc_attr($val), esc_attr($default) ); 
    }
    
    public function render_agency_section_text() {
        // Leitura Direta para garantir o status correto na UI
        global $wpdb;
        $row = $wpdb->get_row( $wpdb->prepare( "SELECT option_value FROM $wpdb->options WHERE option_name = %s", 'wpcm_settings_options' ) );
        $real_options = ( is_object($row) && isset($row->option_value) ) ? maybe_unserialize( $row->option_value ) : array();
        $status = $real_options['agency_connection_status'] ?? 'disconnected';

        if ( $status === 'connected' ) {
            echo '<p>' . esc_html__( 'This site is successfully synced with your WPCM Agency Hub.', 'consent-manager-gdpr-ccpa' ) . '</p>';
        } else {
            echo '<p>' . esc_html__( 'Connect to your Agency Hub below.', 'consent-manager-gdpr-ccpa' ) . '</p>';
        }
    }

    public function render_field_connection_status( $args ) {
        // Leitura Direta para garantir o status correto na UI
        global $wpdb;
        $row = $wpdb->get_row( $wpdb->prepare( "SELECT option_value FROM $wpdb->options WHERE option_name = %s", 'wpcm_settings_options' ) );
        $real_options = ( is_object($row) && isset($row->option_value) ) ? maybe_unserialize( $row->option_value ) : array();
        $status = $real_options['agency_connection_status'] ?? 'disconnected';
        
        if ( $status === 'connected' ) {
            echo '<span style="color: #10b981; font-weight: 600;">' . esc_html__( 'Connected', 'consent-manager-gdpr-ccpa' ) . '</span>';
            echo '<p class="description">' . esc_html__( 'This site is successfully synced with the Hub.', 'consent-manager-gdpr-ccpa' ) . '</p>';
        } else {
            echo '<span style="color: #ef4444; font-weight: 600;">' . esc_html__( 'Disconnected', 'consent-manager-gdpr-ccpa' ) . '</span>';
            echo '<p class="description">' . esc_html__( 'Save your Hub URL and API Key to connect.', 'consent-manager-gdpr-ccpa' ) . '</p>';
        }
        
        echo '<p style="margin-top: 10px;">
                  <a href="#" id="wpcm-verify-connection" class="button button-primary">' . esc_html__( 'Verify Connection', 'consent-manager-gdpr-ccpa' ) . '</a>
                  <span class="spinner" style="float: none; vertical-align: middle;"></span>
              </p>';
        echo '<p id="wpcm-connection-notice" style="margin-top: 10px;"></p>';
    }
}