<?php
/**
 * Service Presets Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gerencia os presets de serviços e sua criação (no Cliente).
 * VERSÃO 3: Presets atualizados com keywords de dependência (ttq, Trustindex)
 */
class WPCM_Service_Presets {

    /**
     * Retorna a lista de todos os presets disponíveis.
     *
     * @return array
     */
    public static function get_all_presets() {
        return array(
            'google_analytics' => array(
                'title'       => 'Google Analytics / GTM',
                'category'    => 'analytics',
                'description' => __( 'Used to analyze website traffic and user behavior.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "google-analytics.com\ngoogletagmanager.com\ngtm.js\nanalytics.js",
                'scripts'     => "",
            ),
            'meta_pixel' => array(
                'title'       => 'Meta (Facebook) Pixel',
                'category'    => 'marketing',
                'description' => __( 'Used to track conversions and build audiences for Meta ads.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "connect.facebook.net\nfbevents.js",
                'scripts'     => "",
            ),
            'google_ads' => array(
                 'title'       => 'Google Ads (Conversion Linker)',
                 'category'    => 'marketing',
                 'description' => __( 'Used to track conversions from Google Ads.', 'consent-manager-gdpr-ccpa' ),
                 'keywords'    => "googleadservices.com\nconversion_async.js",
                 'scripts'     => "",
            ),
            'youtube_vimeo' => array(
                'title'       => 'YouTube / Vimeo Embeds',
                'category'    => 'marketing',
                'description' => __( 'Allows embedded videos from YouTube and Vimeo to load.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "youtube.com\nyoutube-nocookie.com\nvimeo.com\nplayer.vimeo.com",
                'scripts'     => "",
            ),
            
            // *** AQUI ESTÁ A MUDANÇA 1 (TIKTOK) ***
            'tiktok_pixel' => array(
                'title'       => 'TikTok Pixel',
                'category'    => 'marketing',
                'description' => __( 'Used to track visitor actions for TikTok advertising.', 'consent-manager-gdpr-ccpa' ),
                // Adicionamos as keywords que estavam causando o erro
                'keywords'    => "analytics.tiktok.com\ntiktok.com\najaxSnippet.js\nttq.track",
                'scripts'     => "",
            ),
            
            'linkedin' => array(
                'title'       => 'LinkedIn Insight Tag',
                'category'    => 'marketing',
                'description' => __( 'Used to track conversions and build audiences for LinkedIn ads.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "snap.licdn.com\nli.lms-analytics\ninsight.min.js",
                'scripts'     => "",
            ),
            'microsoft_ads' => array(
                'title'       => 'Microsoft (Bing) Ads',
                'category'    => 'marketing',
                'description' => __( 'Used to track conversions for Microsoft Advertising (Bing).', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "bat.bing.com\nbat.js\nuet.js",
                'scripts'     => "",
            ),
            'x_twitter' => array(
                'title'       => 'X (Twitter) Pixel',
                'category'    => 'marketing',
                'description' => __( 'Used to track conversions and build audiences for X ads.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "static.ads-twitter.com\ntwq.js",
                'scripts'     => "",
            ),
            'pinterest' => array(
                'title'       => 'Pinterest Tag',
                'category'    => 'marketing',
                'description' => __( 'Used to track conversions and build audiences for Pinterest ads.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "ads.pinterest.com\npinit.js",
                'scripts'     => "",
            ),
            'hotjar' => array(
                'title'       => 'Hotjar',
                'category'    => 'analytics',
                'description' => __( 'Used for heatmaps, session recordings, and user behavior analytics.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "static.hotjar.com\nhotjar.js",
                'scripts'     => "",
            ),
            
            // *** AQUI ESTÁ A MUDANÇA 2 (TRUSTINDEX) ***
            // Adicionamos um novo preset para o Trustindex
            'trustindex' => array(
                'title'       => 'Trustindex.io Widgets',
                'category'    => 'marketing',
                'description' => __( 'Used to display reviews and social widgets.', 'consent-manager-gdpr-ccpa' ),
                'keywords'    => "cdn.trustindex.io\nTrustindexReadMoreModule.js",
                'scripts'     => "",
            ),
        );
    }

    /**
     * Cria serviços baseados em uma lista de chaves de presets.
     *
     * @param array $preset_keys Lista de chaves dos presets a serem criados (ex: ['google_analytics', 'meta_pixel']).
     * @return int Número de serviços criados com sucesso.
     */
    public static function create_services_from_presets( $preset_keys ) {
        $all_presets = self::get_all_presets();
        $created_count = 0;

        foreach ( $preset_keys as $key ) {
            if ( ! isset( $all_presets[$key] ) ) {
                continue;
            }

            $preset = $all_presets[$key];

            // Verifica se já existe um serviço com este título para evitar duplicatas
            if ( self::service_exists( $preset['title'] ) ) {
                continue;
            }

            $post_data = array(
                'post_title'  => $preset['title'],
                'post_type'   => 'wpcm_service', // CPT do Cliente
                'post_status' => 'publish',
            );

            $post_id = wp_insert_post( $post_data );

            if ( ! is_wp_error( $post_id ) ) {
                update_post_meta( $post_id, '_wpcm_category', $preset['category'] );
                update_post_meta( $post_id, '_wpcm_description', $preset['description'] );
                update_post_meta( $post_id, '_wpcm_keywords', $preset['keywords'] );
                update_post_meta( $post_id, '_wpcm_scripts', $preset['scripts'] );
                $created_count++;
            }
        }

        return $created_count;
    }

    /**
     * Verifica se um serviço com o título dado já existe.
     *
     * @param string $title Título do serviço.
     * @return bool
     */
    private static function service_exists( $title ) {
        $query = new WP_Query( array(
            'post_type'      => 'wpcm_service', // CPT do Cliente
            'post_status'    => 'any',
            'title'          => $title,
            'posts_per_page' => 1,
            'fields'         => 'ids',
        ) );

        return $query->have_posts();
    }

    /**
     * Retorna as assinaturas de arquivos para o Scanner Automático.
     * Mapeia trechos de URL para as chaves dos nossos presets.
     * * @return array
     */
    public static function get_scanner_signatures() {
        return array(
            'googletagmanager.com/gtm.js' => 'google_analytics',
            'google-analytics.com/analytics.js' => 'google_analytics',
            'connect.facebook.net' => 'meta_pixel',
            'fbevents.js' => 'meta_pixel',
            'googleadservices.com' => 'google_ads',
            'analytics.tiktok.com' => 'tiktok_pixel',
            'ttq.load' => 'tiktok_pixel', // inline detection
            'youtube.com/iframe_api' => 'youtube_vimeo',
            'player.vimeo.com' => 'youtube_vimeo',
            'hotjar.com' => 'hotjar',
            'ads-twitter.com' => 'x_twitter',
            'linkedin.com/insight.min.js' => 'linkedin',
            'trustindex.io' => 'trustindex',
        );
    }
}