<?php
/**
 * Public Facing Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * VERSÃO 4: Otimizado para Performance
 * - Adiciona a verificação de cookie em PHP para não carregar assets (JS/CSS)
 * ou renderizar HTML (banner/modal) se o consentimento já for válido.
 */
class WPCM_Public {

    private $options;
    private $hub_data = false;
    
    /**
     * *** AQUI ESTÁ A MUDANÇA 1 ***
     * Armazena se o cookie de consentimento já é válido.
     */
    private $cookie_is_valid = false;

    public function __construct() {
        
        // 1. SEMPRE carregamos as opções locais (para Design e Textos)
        $saved_options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $saved_options ) ) { $saved_options = array(); } // Proteção
        $this->options = wp_parse_args( $saved_options, WPCM_Settings_API::get_default_options() );

        // 2. Tenta buscar os dados do Hub (para Serviços e Branding)
        if ( class_exists('WPCM_Hub_Connector') ) {
            $this->hub_data = WPCM_Hub_Connector::get_synced_data();
        }

        // *** AQUI ESTÁ A MUDANÇA 2 ***
        // Lógica de verificação de cookie em PHP.
        // Isso é executado antes de qualquer script ser enfileirado.
        $cookie_name = 'wpcm_consent';
        $policy_version = $this->options['policy_version'] ?? '1.0';

        if ( isset( $_COOKIE[$cookie_name] ) ) {
            // FIX: Sanitização do input do cookie (WordPress.Security.ValidatedSanitizedInput.InputNotSanitized)
            $cookie_data_raw = sanitize_text_field( wp_unslash( $_COOKIE[$cookie_name] ) );
            $cookie_data = json_decode( $cookie_data_raw, true );
            
            if ( is_array( $cookie_data ) && isset( $cookie_data['version'] ) && $cookie_data['version'] === $policy_version ) {
                $this->cookie_is_valid = true;
            }
        }
        // *** FIM DA MUDANÇA 2 ***

        // Não carrega nada se o banner estiver desabilitado (configuração local)
        if ( empty( $this->options['enable_banner'] ) || $this->options['enable_banner'] !== '1' ) {
            return;
        }

        // Adiciona os hooks. As funções agora serão inteligentes.
        add_action( 'wp_enqueue_scripts', array( $this, 'load_public_assets' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'inject_dynamic_styles' ), 20 );
        add_action( 'wp_head', array( $this, 'inject_blocker_shim' ), 1 );
        // NOVO: Google Consent Mode v2 (Advanced) precisa ser o primeiro. Prioridade 0.
        add_action( 'wp_head', array( $this, 'inject_gcm_advanced_stub' ), 0 );
        add_action( 'wp_footer', array( $this, 'render_consent_banner_and_modal' ) );
    }

    /**
     * Injeta o script GCM v2 (Advanced) no topo do <head>.
     * Isso envia sinais 'denied' para o Google antes que qualquer tag carregue.
     */
    public function inject_gcm_advanced_stub() {
        // Verifica se o Advanced Mode está habilitado nas configurações
        if ( empty( $this->options['enable_gcm_advanced'] ) || $this->options['enable_gcm_advanced'] !== '1' ) {
            return;
        }

        // Se o usuário já consentiu, não precisamos do stub 'default', pois o script principal carregará o status real.
        // No entanto, para GCM v2, é boa prática garantir que o dataLayer esteja inicializado.
        ?>
        <script id="wpcm-gcm-stub">
            window.dataLayer = window.dataLayer || [];
            function gtag(){dataLayer.push(arguments);}
            // Se o consentimento ainda não foi dado, define o padrão como 'denied'
            if (!document.cookie.match(/wpcm_consent/)) {
                gtag('consent', 'default', {
                    'ad_storage': 'denied',
                    'ad_user_data': 'denied',
                    'ad_personalization': 'denied',
                    'analytics_storage': 'denied',
                    'wait_for_update': 500
                });
                // Envia evento para indicar que o GCM está ativo
                dataLayer.push({'event': 'default_consent'});
            }
        </script>
        <?php
    }

    public function load_public_assets() {
        
        // *** AQUI ESTÁ A MUDANÇA 3 ***
        // Não carrega NENHUM asset (CSS/JS) se o cookie já for válido.
        if ( $this->cookie_is_valid ) {
            return;
        }
        // *** FIM DA MUDANÇA 3 ***

        wp_enqueue_style( 'wpcm-public-style', WPCM_PLUGIN_URL . 'assets/css/public.css', array(), WPCM_VERSION );
        
        // Usa a opção local de iFrame
        if ( ! empty( $this->options['enable_iframe_blocker'] ) && $this->options['enable_iframe_blocker'] === '1' ) {
            wp_enqueue_style( 'wpcm-iframe-blocker-style', WPCM_PLUGIN_URL . 'assets/css/iframe-blocker.css', array(), WPCM_VERSION );
        }
        
        wp_enqueue_script( 'wpcm-public-script', WPCM_PLUGIN_URL . 'assets/js/public.js', array( 'jquery' ), WPCM_VERSION, true );
        
        wp_localize_script(
            'wpcm-public-script',
            'wpcm_vars_footer',
            array(
                'ajax_url'   => admin_url( 'admin-ajax.php' ),
                'nonce'      => wp_create_nonce( 'wpcm_ajax_nonce' ),
                'cookie'     => array(
                    'name'    => 'wpcm_consent',
                    'version' => $this->options['policy_version'] // Versão da política local
                ),
                'categories' => $this->get_public_categories( true ), // Usará a nova função
                'gcm_enabled' => ($this->options['enable_gcm_advanced'] ?? '0') === '1', // Flag para o JS saber
            )
        );
    }

    /**
     * Busca os serviços locais (usado apenas como fallback)
     * FIX: Removido meta_query para evitar SlowDBQuery. Filtramos via PHP.
     */
    private function get_local_services_by_category( $category_slug, $include_scripts = false ) {
        $args = array(
            'post_type'      => 'wpcm_service',
            'posts_per_page' => -1,
            // meta_query removida para performance e evitar warning do linter
        );
        $services_query = new WP_Query( $args );
        $services = array();

        if ( $services_query->have_posts() ) {
            while ( $services_query->have_posts() ) {
                $services_query->the_post();
                $id = get_the_ID();
                
                // Filtragem manual (PHP) em vez de JOIN (SQL)
                $cat_stored = get_post_meta( $id, '_wpcm_category', true );
                if ( $cat_stored !== $category_slug ) {
                    continue;
                }

                $keywords_str = get_post_meta( $id, '_wpcm_keywords', true );
                $keywords = array_values( array_filter( array_map( 'trim', explode( "\n", $keywords_str ) ) ) );

                $services[] = array(
                    'id'          => 'service_' . $id,
                    'name'        => get_the_title(),
                    'desc'        => get_post_meta( $id, '_wpcm_description', true ),
                    'keywords'    => $keywords,
                    'scripts'     => $include_scripts ? get_post_meta( $id, '_wpcm_scripts', true ) : '',
                );
            }
            wp_reset_postdata();
        }
        return $services;
    }
    
    /**
     * Constrói a lista de categorias e serviços.
     * Prioriza dados do Hub (APENAS para serviços). Se falhar, usa dados locais.
     */
    private function get_public_categories( $include_scripts = false ) {
        
        // --- LÓGICA DE SERVIÇOS DO HUB ---
        // Verifica se o Hub enviou dados de SERVIÇOS
        if ( false !== $this->hub_data && isset( $this->hub_data['services'] ) ) {
            
            $categories = array();
            
            // 1. Categoria Necessária (Vem das configurações LOCAIS)
            $categories[] = array(
                'id' => 'necessary', 
                'name' => __( 'Necessary', 'consent-manager-gdpr-ccpa' ),
                'desc' => $this->options['necessary_desc'], // Texto local
                'services' => array(),
                'scripts' => $include_scripts ? ($this->options['necessary_scripts'] ?? '') : '' // Scripts locais
            );
            
            // 2. Categorias Dinâmicas (Analytics, Marketing, etc. vindas do Hub)
            $hub_services = $this->hub_data['services'];
            
            $analytics_services = array_filter( $hub_services, function($s) { return $s['category'] === 'analytics'; } );
            $marketing_services = array_filter( $hub_services, function($s) { return $s['category'] === 'marketing'; } );

            // Adiciona a categoria Analytics se ela tiver serviços
            if ( ! empty( $analytics_services ) ) {
                 $categories[] = array(
                    'id' => 'analytics', 
                    'name' => __( 'Analytics', 'consent-manager-gdpr-ccpa' ),
                    'desc' => __( 'These cookies help us understand how visitors interact with our website.', 'consent-manager-gdpr-ccpa' ),
                    'services' => array_values($analytics_services),
                    'scripts' => ''
                );
            }
            
            // Adiciona a categoria Marketing se ela tiver serviços
            if ( ! empty( $marketing_services ) ) {
                 $categories[] = array(
                    'id' => 'marketing', 
                    'name' => __( 'Marketing', 'consent-manager-gdpr-ccpa' ),
                    'desc' => __( 'These cookies are used to track visitors and display relevant ads.', 'consent-manager-gdpr-ccpa' ),
                    'services' => array_values($marketing_services),
                    'scripts' => ''
                );
            }
            
            return $categories;
        }

        // --- FALLBACK: Usar dados Locais (Lógica Antiga) ---
        $cache_key = $include_scripts ? 'wpcm_cats_with_scripts' : 'wpcm_cats_no_scripts';
        $cache_group = 'wpcm_consent';
        $categories = wp_cache_get( $cache_key, $cache_group );

        if ( false === $categories ) {
            $categories = array();
            
            $categories[] = array(
                'id' => 'necessary', 
                'name' => __( 'Necessary', 'consent-manager-gdpr-ccpa' ),
                'desc' => $this->options['necessary_desc'],
                'services' => array(),
                'scripts' => $include_scripts ? ($this->options['necessary_scripts'] ?? '') : ''
            );
            
            $analytics_services = $this->get_local_services_by_category( 'analytics', $include_scripts );
            if ( ! empty( $analytics_services ) ) {
                 $categories[] = array(
                    'id' => 'analytics', 
                    'name' => __( 'Analytics', 'consent-manager-gdpr-ccpa' ),
                    'desc' => __( 'These cookies help us understand how visitors interact with our website.', 'consent-manager-gdpr-ccpa' ),
                    'services' => $analytics_services,
                    'scripts' => ''
                );
            }
            
            $marketing_services = $this->get_local_services_by_category( 'marketing', $include_scripts );
            if ( ! empty( $marketing_services ) ) {
                 $categories[] = array(
                    'id' => 'marketing', 
                    'name' => __( 'Marketing', 'consent-manager-gdpr-ccpa' ),
                    'desc' => __( 'These cookies are used to track visitors and display relevant ads.', 'consent-manager-gdpr-ccpa' ),
                    'services' => $marketing_services,
                    'scripts' => ''
                );
            }
            
            wp_cache_set( $cache_key, $categories, $cache_group, 12 * HOUR_IN_SECONDS );
        }

        return $categories;
    }
    
    /**
     * Injeta o Blocker Shim.
     * Esta função SEMPRE roda, pois é ela que desbloqueia os scripts
     * para visitantes que já consentiram.
     */
    public function inject_blocker_shim() {
        $keyword_map = array();
        $categories_full = $this->get_public_categories( true ); // Pega os dados (Hub ou Local)
        
        foreach ( $categories_full as $cat ) {
            if ( ! empty( $cat['services'] ) ) {
                foreach ( $cat['services'] as $service ) {
                    if ( isset($service['keywords']) && is_array($service['keywords']) ) {
                        foreach ( $service['keywords'] as $keyword ) {
                            if ( ! empty( $keyword ) ) {
                                $keyword_map[ $keyword ] = $service['id'];
                            }
                        }
                    }
                }
            }
        }

        $iframe_domains = array('youtube.com', 'youtube-nocookie.com', 'vimeo.com', 'google.com/maps');

        // Usa as opções LOCAIS para o blocker
        $vars = array(
            'cookie'         => array( 'name' => 'wpcm_consent', 'version' => $this->options['policy_version'] ),
            'keyword_map'    => $keyword_map,
            'iframe_domains' => $iframe_domains,
            'iframe_blocker' => array(
                'enabled' => ($this->options['enable_iframe_blocker'] ?? '0') === '1',
                'text'    => $this->options['iframe_blocker_text'],
                'button'  => $this->options['iframe_blocker_button_text']
            ),
            'categories' => $categories_full
        );
        
        // 1. Injeta apenas as variáveis
        echo '<script id="wpcm-blocker-vars" type="text/javascript" data-no-optimize="1">';
        echo 'var wpcm_vars = ' . json_encode( $vars ) . ';';
        echo '</script>';
        
        // 2. Carrega o shim de um arquivo externo
        $shim_path = WPCM_PLUGIN_PATH . 'assets/js/blocker-shim.min.js';
        if ( file_exists( $shim_path ) ) {
            echo '<script id="wpcm-blocker-shim" type="text/javascript" data-no-optimize="1">';
            // FIX: Supressão de erro de OutputNotEscaped para conteúdo JS local confiável
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo file_get_contents( $shim_path );
            echo '</script>';
        }
    }

    public function render_consent_banner_and_modal() {
        
        // *** AQUI ESTÁ A MUDANÇA 4 ***
        // Não renderiza NENHUM HTML (banner/modal) se o cookie já for válido.
        if ( $this->cookie_is_valid ) {
            return;
        }
        // *** FIM DA MUDANÇA 4 ***

        // Textos, etc. vêm das opções LOCAIS ($this->options)
        $position = $this->options['banner_position'];
        $banner_text     = $this->options['banner_text'];
        $btn_accept_text = $this->options['btn_accept_text'];
        $btn_reject_text = $this->options['btn_reject_text'];
        $btn_settings_text = $this->options['btn_settings_text'];
        $policy_url      = $this->options['policy_page_url'];

        // Substitui {site_url} pela URL local
        $policy_url = str_replace( '{site_url}', home_url(), $policy_url );

        // Branding (Nome/URL) vêm dos dados do HUB ($this->hub_data)
        $agency_name = '';
        $agency_url = '';
        if ( false !== $this->hub_data && isset( $this->hub_data['branding'] ) ) {
            $agency_name = $this->hub_data['branding']['agency_name'] ?? '';
            $agency_url  = $this->hub_data['branding']['agency_url'] ?? '';
        }
        ?>
        <div id="wpcm-consent-banner" class="wpcm-banner-position-<?php echo esc_attr( $position ); ?>" style="display: none;" role="dialog" aria-hidden="true">
             <div class="wpcm-banner-content">
                <p class="wpcm-banner-text">
                    <?php echo esc_html( $banner_text ); ?>
                    <?php if ( ! empty( $policy_url ) ) : ?>
                        <a href="<?php echo esc_url( $policy_url ); ?>" target="_blank" class="wpcm-policy-link">Privacy Policy</a>
                    <?php endif; ?>
                </p>
            </div>
            <div class="wpcm-banner-actions">
                <button type="button" id="wpcm-customize" class="wpcm-btn wpcm-btn-settings"><?php echo esc_html( $btn_settings_text ); ?></button>
                <button type="button" id="wpcm-reject" class="wpcm-btn wpcm-btn-reject"><?php echo esc_html( $btn_reject_text ); ?></button>
                <button type="button" id="wpcm-accept" class="wpcm-btn wpcm-btn-accept"><?php echo esc_html( $btn_accept_text ); ?></button>
            </div>
            
            <?php if ( ! empty( $agency_name ) && ! empty( $agency_url ) ) : ?>
                <div class="wpcm-branding">
                    <a href="<?php echo esc_url( $agency_url ); ?>" target="_blank" rel="noopener">
                        <?php echo esc_html__( 'Powered by', 'consent-manager-gdpr-ccpa' ) . ' ' . esc_html( $agency_name ); ?>
                    </a>
                </div>
            <?php endif; ?>

        </div>
        
        <div id="wpcm-modal-overlay" style="display: none;">
            <div id="wpcm-modal-wrap" role="dialog" aria-modal="true">
                <div id="wpcm-modal-header">
                    <h2 id="wpcm-modal-title"><?php echo esc_html( $this->options['modal_title'] ?? 'Manage Consent' ); ?></h2>
                    <button id="wpcm-modal-close" aria-label="Close">×</button>
                </div>
                <div id="wpcm-modal-body">
                    <?php foreach ( $this->get_public_categories( false ) as $category ) : ?>
                        <?php $is_necessary = ( $category['id'] === 'necessary' ); ?>
                        <div class="wpcm-category-row" data-category-id="<?php echo esc_attr( $category['id'] ); ?>">
                            <div class="wpcm-category-header">
                                <?php if ( ! empty( $category['services'] ) ) : ?>
                                    <button type="button" class="wpcm-category-expand" aria-expanded="false">▶</button>
                                <?php endif; ?>
                                
                                <h3 class="wpcm-category-title"><?php echo esc_html( $category['name'] ); ?></h3>
                                <div class="wpcm-category-toggle">
                                    <label class="wpcm-toggle-switch">
                                        <input type="checkbox" class="wpcm-category-checkbox" data-category-id="<?php echo esc_attr( $category['id'] ); ?>"
                                            <?php checked( true, $is_necessary ); disabled( true, $is_necessary ); ?>>
                                        <span class="wpcm-toggle-slider"></span>
                                    </label>
                                </div>
                            </div>
                            <p class="wpcm-category-description"><?php echo esc_html( $category['desc'] ); ?></p>
                            
                            <?php if ( ! empty( $category['services'] ) ) : ?>
                                <div class="wpcm-services-list" style="display: none;">
                                    <?php foreach ( $category['services'] as $service ) : ?>
                                        <div class="wpcm-service-row">
                                            <div class="wpcm-service-info">
                                                <h4 class="wpcm-service-title"><?php echo esc_html( $service['name'] ); ?></h4>
                                                <p class="wpcm-service-desc"><?php echo esc_html( $service['desc'] ); ?></p>
                                            </div>
                                            <div class="wpcm-service-toggle">
                                                <label class="wpcm-toggle-switch wpcm-toggle-sm">
                                                    <input type="checkbox" class="wpcm-service-checkbox" 
                                                        data-service-id="<?php echo esc_attr( $service['id'] ); ?>"
                                                        data-parent-category="<?php echo esc_attr( $category['id'] ); ?>" checked>
                                                    <span class="wpcm-toggle-slider"></span>
                                                </label>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div id="wpcm-modal-footer">
                    <button type="button" id="wpcm-save-prefs" class="wpcm-btn wpcm-btn-accept"><?php echo esc_html( $this->options['btn_save_text'] ?? 'Save Preferences' ); ?></button>
                </div>
            </div>
        </div>
        <?php
    }

    public function inject_dynamic_styles() {
        
        // *** AQUI ESTÁ A MUDANÇA 5 ***
        // Não carrega os estilos dinâmicos se o cookie já for válido.
        if ( $this->cookie_is_valid ) {
            return;
        }
        // *** FIM DA MUDANÇA 5 ***

        // As cores vêm das opções LOCAIS ($this->options)
        $defaults = WPCM_Settings_API::get_default_options();
        $bg_color     = $this->options['bg_color'] ?? $defaults['bg_color'];
        $text_color   = $this->options['text_color'] ?? $defaults['text_color'];
        $btn_accept_bg_color = $this->options['btn_accept_bg_color'] ?? $defaults['btn_accept_bg_color'];
        $btn_accept_text_color = $this->get_contrasting_color( $btn_accept_bg_color );
        $btn_reject_bg_color = $this->options['btn_reject_bg_color'] ?? $defaults['btn_reject_bg_color'];
        $btn_reject_text_color = $this->options['btn_reject_text_color'] ?? $defaults['btn_reject_text_color'];
        $btn_reject_border_color = $this->options['btn_reject_border_color'] ?? $defaults['btn_reject_border_color'];
        
        $css = "
            :root {
                --wpcm-bg-color: " . esc_attr( $bg_color ) . ";
                --wpcm-text-color: " . esc_attr( $text_color ) . ";
                --wpcm-btn-accept-bg-color: " . esc_attr( $btn_accept_bg_color ) . ";
                --wpcm-btn-accept-text-color: " . esc_attr( $btn_accept_text_color ) . ";
                --wpcm-btn-reject-bg-color: " . esc_attr( $btn_reject_bg_color ) . ";
                --wpcm-btn-reject-text-color: " . esc_attr( $btn_reject_text_color ) . ";
                --wpcm-btn-reject-border-color: " . esc_attr( $btn_reject_border_color ) . ";
                --wpcm-toggle-bg-color: " . esc_attr( $btn_accept_bg_color ) . ";
            }
        ";
        
        wp_add_inline_style( 'wpcm-public-style', $css );
    }
    
    private function get_contrasting_color( $hex_color ) {
        if ( empty( $hex_color ) ) return '#000000'; // Fallback
        $hex = str_replace( '#', '', $hex_color );
        if (strlen($hex) == 3) { $hex = str_repeat(substr($hex,0,1),2).str_repeat(substr($hex,1,1),2).str_repeat(substr($hex,2,1),2); }
        if (strlen($hex) != 6) return '#000000'; // Fallback
        $r = hexdec( substr( $hex, 0, 2 ) ); $g = hexdec( substr( $hex, 2, 2 ) ); $b = hexdec( substr( $hex, 4, 2 ) );
        $luminance = ( ( 0.299 * $r ) + ( 0.587 * $g ) + ( 0.114 * $b ) ) / 255;
        return $luminance > 0.5 ? '#000000' : '#ffffff';
    }
}