<?php
/**
 * Log Sync Scheduler Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gerencia a tarefa agendada (WP-Cron) para enviar lotes de logs ao Hub.
 * * @package Consent_Manager_GDPR_CCPA
 */
class WPCM_Log_Sync_Scheduler {

    /**
     * O nome do nosso hook de cron.
     */
    const CRON_HOOK = 'wpcm_sync_logs_event';

    /**
     * O nome do "lock" (transient) para evitar corridas.
     */
    const LOCK_TRANSIENT = 'wpcm_sync_logs_lock';

    /**
     * Namespace da API do Hub (para onde vamos enviar).
     */
    protected $hub_api_namespace = 'wpcm-hub/v1';

    public function __construct() {
        // Adiciona o gatilho para a função que roda a tarefa
        add_action( self::CRON_HOOK, array( $this, 'run_sync_task' ) );
    }

    /**
     * Agenda o evento cron (chamado na ativação/inicialização).
     */
    public static function schedule_cron_event() {
        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            // Agenda para rodar a cada hora
            wp_schedule_event( time(), 'hourly', self::CRON_HOOK );
        }
    }

    /**
     * Limpa o evento cron (chamado na desativação).
     */
    public static function clear_cron_event() {
        wp_clear_scheduled_hook( self::CRON_HOOK );
    }

    /**
     * A função principal que é executada pelo WP-Cron.
     */
    public function run_sync_task() {
        global $wpdb;

        // 1. Verificar se o Hub está configurado
        $options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $options ) ) $options = array();

        $hub_url = $options['agency_hub_url'] ?? '';
        $api_key = $options['agency_api_key'] ?? '';
        $status  = $options['agency_connection_status'] ?? 'disconnected';

        if ( 'connected' !== $status || empty( $hub_url ) || empty( $api_key ) ) {
            return; // Não faz nada se não estiver conectado
        }

        // 2. Definir um "Lock" para evitar execução duplicada
        // Se o 'lock' já existir, significa que uma tarefa já está rodando.
        if ( get_transient( self::LOCK_TRANSIENT ) ) {
            return;
        }
        // Define o 'lock' por 10 minutos (tempo máximo da tarefa)
        set_transient( self::LOCK_TRANSIENT, 1, 10 * MINUTE_IN_SECONDS );

        
        // 3. Buscar um lote de logs não sincronizados
        $table_name = $wpdb->prefix . 'consent_log';
        $batch_size = 100; // Envia 100 logs de cada vez

        // ATENÇÃO: Adicionado 'ip_address' para garantir envio do dado coletado
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery
        $logs_to_sync = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, user_hash, timestamp, consent_categories, policy_version, ip_address 
                 FROM $table_name 
                 WHERE synced_to_hub = 0 
                 ORDER BY id ASC 
                 LIMIT %d",
                $batch_size
            )
        );
        // phpcs:enable

        if ( empty( $logs_to_sync ) ) {
            delete_transient( self::LOCK_TRANSIENT ); // Libera o lock
            return; // Nada para fazer
        }

        // 4. Enviar o lote para a API do Hub
        $api_endpoint = trailingslashit( $hub_url ) . 'wp-json/' . $this->hub_api_namespace . '/sync_logs';
        
        $response = wp_remote_post( $api_endpoint, array(
            'method'      => 'POST',
            'timeout'     => 30, // 30 segundos para lidar com lotes grandes
            'headers'     => array(
                'X-Api-Key'    => $api_key, // Autenticação
                'Content-Type' => 'application/json',
            ),
            'body'        => wp_json_encode( $logs_to_sync ), // Envia o lote como JSON
            'user-agent'  => 'WPCM-Client/' . ( defined( 'WPCM_VERSION' ) ? WPCM_VERSION : '1.0' ),
        ) );

        // 5. Lidar com a Resposta
        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
            // Sucesso! O Hub recebeu os logs.
            
            // Pega os IDs dos logs que acabamos de enviar
            $synced_ids = wp_list_pluck( $logs_to_sync, 'id' );
            $id_placeholders = implode( ',', array_fill( 0, count( $synced_ids ), '%d' ) );

            // Marca os logs como "sincronizados" no banco de dados local.
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $wpdb->query(
                $wpdb->prepare(
                    "UPDATE $table_name SET synced_to_hub = 1 WHERE id IN ( $id_placeholders )",
                    $synced_ids
                )
            );
            // phpcs:enable
        }
        
        // 6. Liberar o "Lock"
        delete_transient( self::LOCK_TRANSIENT );
    }
}