<?php
/**
 * Ajax Handler Class - Versão Standard WP Cache Buster
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPCM_Ajax_Handler {

    protected $hub_api_namespace = 'wpcm-hub/v1';

    public function __construct() {
        add_action( 'wp_ajax_wpcm_log_consent', array( $this, 'log_consent' ) );
        add_action( 'wp_ajax_nopriv_wpcm_log_consent', array( $this, 'log_consent' ) );
        add_action( 'wp_ajax_wpcm_verify_hub_connection', array( $this, 'verify_hub_connection' ) );
    }

    public function log_consent() {
        check_ajax_referer( 'wpcm_ajax_nonce', 'nonce' );
        $consent_data_json = isset( $_POST['consent_data'] ) ? sanitize_textarea_field( wp_unslash( $_POST['consent_data'] ) ) : '';
        $consent_data = json_decode( $consent_data_json, true );

        if ( ! $consent_data || ! isset( $consent_data['version'] ) || ! isset( $consent_data['categories'] ) ) {
            wp_send_json_error( 'Invalid data' );
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'consent_log';
        
        $user_ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        $user_hash = hash( 'sha256', $user_ip . $user_agent );

        $options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $options ) ) $options = array();
        
        $enable_ip_logging = $options['enable_ip_logging'] ?? '0';
        $ip_to_store = ''; 
        if ( $enable_ip_logging === '1' ) { $ip_to_store = $user_ip; }
        
        $data_to_insert = array(
            'user_hash'          => $user_hash,
            'timestamp'          => current_time( 'mysql', 1 ),
            'consent_categories' => implode( ',', array_map('sanitize_text_field', $consent_data['categories']) ),
            'policy_version'     => sanitize_text_field( $consent_data['version'] ),
            'ip_address'         => $ip_to_store,
        );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert( $table_name, $data_to_insert );

        if ( $result ) {
            if ( class_exists( 'WPCM_Core' ) ) { WPCM_Core::clear_log_cache(); }
            wp_send_json_success( 'Consent logged' );
        } else {
            wp_send_json_error( 'Failed to log consent' );
        }
    }

    public function verify_hub_connection() {
        check_ajax_referer( 'wpcm_ajax_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'consent-manager-gdpr-ccpa' ) ) );
            return;
        }

        $hub_url = isset( $_POST['hub_url'] ) ? esc_url_raw( wp_unslash( $_POST['hub_url'] ) ) : '';
        $api_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';

        $clean_hub_url  = rtrim( preg_replace( '(^https?://)', '', $hub_url ), '/' );
        $clean_site_url = rtrim( preg_replace( '(^https?://)', '', get_site_url() ), '/' );
        $clean_home_url = rtrim( preg_replace( '(^https?://)', '', home_url() ), '/' );

        if ( class_exists( 'WPCM_Hub_Core' ) && ( $clean_hub_url === $clean_site_url || $clean_hub_url === $clean_home_url ) ) {
            $this->handle_local_connection( $api_key, $hub_url );
        } else {
            $this->handle_remote_connection( $hub_url, $api_key );
        }
    }

    private function handle_local_connection( $api_key, $hub_url ) {
        $args = array(
            'post_type' => 'wpcm_hub_site', 'post_status' => 'publish', 'posts_per_page' => 1,
            'meta_query' => array( array( 'key' => '_api_key', 'value' => $api_key, 'compare' => '=' ) ),
        );
        $site_query = new WP_Query( $args );

        if ( ! $site_query->have_posts() ) {
            wp_send_json_error( array( 'message' => __( 'Invalid API Key for local connection.', 'consent-manager-gdpr-ccpa' ) ) );
            return;
        }
        
        $site_post_id = $site_query->posts[0]->ID;
        $site_title = $site_query->posts[0]->post_title;

        update_post_meta( $site_post_id, '_site_url', get_site_url() );
        update_post_meta( $site_post_id, '_status', 'connected' );

        $this->save_connection_settings( 'connected', $hub_url, $api_key );

        wp_send_json_success( array(
            'message'   => __( 'Successfully connected to local Hub!', 'consent-manager-gdpr-ccpa' ),
            'site_name' => $site_title . ' (Local)',
        ) );
    }

    private function handle_remote_connection( $hub_url, $api_key ) {
        if ( empty( $hub_url ) || empty( $api_key ) ) {
            wp_send_json_error( array( 'message' => __( 'Hub URL and API Key are required.', 'consent-manager-gdpr-ccpa' ) ) );
            return;
        }

        $api_endpoint = trailingslashit( $hub_url ) . 'wp-json/' . $this->hub_api_namespace . '/connect';
        $body_args = array( 'api_key'  => $api_key, 'site_url' => get_site_url() );

        $response = wp_remote_post( $api_endpoint, array(
            'method'      => 'POST',
            'timeout'     => 15,
            'body'        => $body_args,
            'user-agent'  => 'WPCM-Client/' . ( defined( 'WPCM_VERSION' ) ? WPCM_VERSION : '1.0' ),
            'sslverify'   => false, 
        ) );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => 'Connection Error: ' . $response->get_error_message() ) );
            return;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( $response_code === 200 && isset( $data['connected'] ) && $data['connected'] === true ) {
            
            $this->save_connection_settings( 'connected', $hub_url, $api_key );

            if ( class_exists('WPCM_Hub_Connector') ) {
                WPCM_Hub_Connector::clear_cache();
                WPCM_Hub_Connector::get_synced_data(); 
            }
            $this->purge_caches();

            wp_send_json_success( array(
                'message'   => __( 'Successfully connected to Hub!', 'consent-manager-gdpr-ccpa' ),
                'site_name' => $data['site_name'] ?? 'Agency Hub',
            ) );

        } else {
            $this->save_connection_settings( 'disconnected', $hub_url, $api_key );
            
            $error_message = isset( $data['error'] ) ? $data['error'] : __( 'Unknown error from Hub.', 'consent-manager-gdpr-ccpa' );
            wp_send_json_error( array( 'message' => $error_message ) );
        }
    }

    /**
     * Salva configurações garantindo que o cache seja invalidado.
     */
    private function save_connection_settings( $status, $url, $key ) {
        
        wp_cache_delete( 'wpcm_settings_options', 'options' );
        $options = get_option( 'wpcm_settings_options', array() );
        if ( ! is_array( $options ) ) $options = array();

        $options['agency_connection_status'] = $status;
        $options['agency_hub_url']           = $url;
        $options['agency_api_key']           = $key;
        
        // DELETE + UPDATE: Método padrão para forçar atualização de cache
        delete_option( 'wpcm_settings_options' );
        update_option( 'wpcm_settings_options', $options );
        
        return true;
    }

    private function purge_caches() {
        wp_cache_flush();
        if ( class_exists( 'LiteSpeed_Cache_API' ) ) { LiteSpeed_Cache_API::purge_all(); }
        if ( function_exists( 'w3tc_flush_all' ) ) { w3tc_flush_all(); }
    }
}