<?php
/**
 * White Label Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gerencia toda a lógica de White-Labeling.
 * Esta classe é carregada imediatamente (antes do 'init') 
 * para garantir que os filtros 'all_plugins' sejam aplicados a tempo.
 */
class WPCM_White_Label {

    /**
     * Armazena os dados do Hub para evitar buscas repetidas.
     * @var array|bool
     */
    private $whitelabel_data = null;

    public function __construct() {
        // Estes filtros rodam antes do 'init', por isso esta classe
        // deve ser instanciada imediatamente.
        add_filter( 'all_plugins', array( $this, 'filter_plugin_data' ) );
        add_filter( 'plugin_row_meta', array( $this, 'filter_plugin_row_meta' ), 10, 2 );

        // Este hook roda depois do 'init', então ele busca os dados
        // que já foram carregados pelo 'load_data'.
        add_action( 'admin_menu', array( $this, 'hide_plugin_menu' ), 999 );
    }

    /**
     * Busca e armazena os dados de White-Label do Hub.
     */
    private function load_data() {
        // Se já buscamos, não busca de novo.
        if ( null !== $this->whitelabel_data ) {
            return;
        }

        $this->whitelabel_data = false; // Define um padrão
        if ( class_exists('WPCM_Hub_Connector') ) {
            $data = WPCM_Hub_Connector::get_synced_data();
            if ( false !== $data && isset( $data['whitelabel'] ) ) {
                $this->whitelabel_data = $data['whitelabel'];
            }
        }
    }

    /**
     * Callback: Filtra a lista de plugins para mudar o nome/descrição.
     */
    public function filter_plugin_data( $plugins ) {
        // Busca os dados do Hub (do cache/transient)
        $this->load_data();

        // Se não houver dados de White-Label, não faz nada
        if ( empty( $this->whitelabel_data ) || empty( $this->whitelabel_data['name'] ) ) {
            return $plugins;
        }

        // WPCM_PLUGIN_BASENAME é definido no 'wp-consent-manager.php'
        if ( isset( $plugins[ WPCM_PLUGIN_BASENAME ] ) ) {
            
            // Sobrescreve o Nome
            $plugins[ WPCM_PLUGIN_BASENAME ]['Name'] = $this->whitelabel_data['name'];
            
            // Sobrescreve a Descrição (se não estiver vazia)
            if ( ! empty( $this->whitelabel_data['desc'] ) ) {
                $plugins[ WPCM_PLUGIN_BASENAME ]['Description'] = $this->whitelabel_data['desc'];
            }
            
            // Remove os dados do autor original
            $plugins[ WPCM_PLUGIN_BASENAME ]['Author'] = '';
            $plugins[ WPCM_PLUGIN_BASENAME ]['AuthorURI'] = '';
            $plugins[ WPCM_PLUGIN_BASENAME ]['PluginURI'] = '';
        }
        return $plugins;
    }

    /**
     * Callback: Remove os links "Visitar site do plugin" e "Ver detalhes"
     */
    public function filter_plugin_row_meta( $links, $file ) {
        // Busca os dados (só por garantia, mas 'filter_plugin_data' já deve ter rodado)
        $this->load_data();

        // Se não houver dados de White-Label, não faz nada
        if ( empty( $this->whitelabel_data ) || empty( $this->whitelabel_data['name'] ) ) {
            return $links;
        }

        if ( $file === WPCM_PLUGIN_BASENAME ) {
            unset( $links[2] ); // Remove "Visit plugin site"
            unset( $links['view-details'] ); // Remove "View details"
        }
        return $links;
    }

    /**
     * Callback: Esconde o menu do plugin no painel do cliente.
     */
    public function hide_plugin_menu() {
        // Busca os dados
        $this->load_data();
        
        // Hook 1: Esconder o menu de administração
        if ( ! empty( $this->whitelabel_data['hide'] ) && $this->whitelabel_data['hide'] === '1' ) {
            // 'wpcm_settings' é o slug do menu principal em class-admin-menu.php
            remove_menu_page( 'wpcm_settings' );
        }
    }
}