<?php
/**
 * Custom Post Type (Services) Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPCM_CPT_Services {

    public function __construct() {
        add_action( 'init', array( $this, 'register_post_type' ) );
        add_action( 'add_meta_boxes', array( $this, 'add_metabox' ) );
        add_action( 'save_post_wpcm_service', array( $this, 'save_metabox_and_clear_cache' ) );
        add_action( 'delete_post', array( $this, 'clear_cache_on_delete' ) );
    }

    public function register_post_type() {
        $labels = array(
            'name'               => __( 'Consent Services', 'consent-manager-gdpr-ccpa' ),
            'singular_name'      => __( 'Service', 'consent-manager-gdpr-ccpa' ),
            'add_new'            => __( 'Add New Service', 'consent-manager-gdpr-ccpa' ),
            'add_new_item'       => __( 'Add New Service', 'consent-manager-gdpr-ccpa' ),
            'edit_item'          => __( 'Edit Service', 'consent-manager-gdpr-ccpa' ),
            'new_item'           => __( 'New Service', 'consent-manager-gdpr-ccpa' ),
            'all_items'          => __( 'All Services', 'consent-manager-gdpr-ccpa' ),
            'view_item'          => __( 'View Service', 'consent-manager-gdpr-ccpa' ),
            'search_items'       => __( 'Search Services', 'consent-manager-gdpr-ccpa' ),
            'not_found'          => __( 'No services found', 'consent-manager-gdpr-ccpa' ),
            'not_found_in_trash' => __( 'No services found in Trash', 'consent-manager-gdpr-ccpa' ),
            'menu_name'          => __( 'Services', 'consent-manager-gdpr-ccpa' )
        );

        $args = array(
            'labels'              => $labels,
            'public'              => false,
            'publicly_queryable'  => false,
            'show_ui'             => true,
            'show_in_menu'        => false,
            'query_var'           => false,
            'rewrite'             => false,
            'capability_type'     => 'post',
            'has_archive'         => false,
            'hierarchical'        => false,
            'menu_position'       => null,
            'supports'            => array( 'title' ),
            'menu_icon'           => 'dashicons-admin-network',
        );

        register_post_type( 'wpcm_service', $args );
    }

    public function add_metabox() {
        add_meta_box(
            'wpcm_service_settings',
            __( 'Service Settings', 'consent-manager-gdpr-ccpa' ),
            array( $this, 'render_metabox' ),
            'wpcm_service',
            'normal',
            'high'
        );
    }

    public function render_metabox( $post ) {
        wp_nonce_field( 'wpcm_service_save', 'wpcm_service_nonce' );
        $category  = get_post_meta( $post->ID, '_wpcm_category', true );
        $desc      = get_post_meta( $post->ID, '_wpcm_description', true );
        $keywords  = get_post_meta( $post->ID, '_wpcm_keywords', true );
        $scripts   = get_post_meta( $post->ID, '_wpcm_scripts', true );

        ?>
        <table class="form-table">
            <tbody>
                <tr>
                    <th><label for="wpcm_category"><?php esc_html_e( 'Category', 'consent-manager-gdpr-ccpa' ); ?></label></th>
                    <td>
                        <select name="wpcm_category" id="wpcm_category" required>
                            <option value="analytics" <?php selected( $category, 'analytics' ); ?>><?php esc_html_e( 'Analytics', 'consent-manager-gdpr-ccpa' ); ?></option>
                            <option value="marketing" <?php selected( $category, 'marketing' ); ?>><?php esc_html_e( 'Marketing', 'consent-manager-gdpr-ccpa' ); ?></option>
                            </select>
                        <p class="description"><?php esc_html_e( 'Assign this service to a category.', 'consent-manager-gdpr-ccpa' ); ?></p>
                    </td>
                </tr>
                <tr>
                    <th><label for="wpcm_description"><?php esc_html_e( 'Service Description', 'consent-manager-gdpr-ccpa' ); ?></label></th>
                    <td>
                        <textarea name="wpcm_description" id="wpcm_description" class="large-text" rows="3"><?php echo esc_textarea( $desc ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'This text will be shown to the user in the consent modal.', 'consent-manager-gdpr-ccpa' ); ?></p>
                    </td>
                </tr>
                <tr>
                    <th><label for="wpcm_keywords"><?php esc_html_e( 'Auto-Blocker Keywords', 'consent-manager-gdpr-ccpa' ); ?></label></th>
                    <td>
                        <textarea name="wpcm_keywords" id="wpcm_keywords" class="large-text" rows="3"><?php echo esc_textarea( $keywords ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'One keyword/domain per line (e.g., "googletagmanager.com"). This tells the shim blocker which scripts belong to this service.', 'consent-manager-gdpr-ccpa' ); ?></p>
                    </td>
                </tr>
                <tr>
                    <th><label for="wpcm_scripts"><?php esc_html_e( 'Manual Scripts (Fallback)', 'consent-manager-gdpr-ccpa' ); ?></label></th>
                    <td>
                        <textarea name="wpcm_scripts" id="wpcm_scripts" class="large-text" rows="5" style="font-family: Consolas, Monaco, monospace;"><?php echo esc_textarea( $scripts ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'Paste scripts here as a fallback. These will be injected ONLY if this specific service is consented to.', 'consent-manager-gdpr-ccpa' ); ?></p>
                    </td>
                </tr>
            </tbody>
        </table>
        <?php
    }

    private function _get_allowed_script_html() {
        return array(
            'script' => array(
                'src'     => true,
                'type'    => true,
                'async'   => true,
                'defer'   => true,
                'charset' => true,
                'id'      => true,
                'class'   => true,
            ),
        );
    }

    private function _sanitize_scripts_field( $input ) {
        return wp_kses( $input, $this->_get_allowed_script_html() );
    }

    public function save_metabox_and_clear_cache( $post_id ) {
        // FIX: Adicionado wp_unslash para o nonce
        $nonce = isset( $_POST['wpcm_service_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['wpcm_service_nonce'] ) ) : '';
        
        if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wpcm_service_save' ) ) {
            return;
        }
        
        if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || wp_is_post_revision( $post_id ) ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $fields_to_sanitize = array(
            'wpcm_category'    => 'sanitize_text_field',
            'wpcm_description' => 'sanitize_textarea_field',
            'wpcm_keywords'    => 'sanitize_textarea_field',
        );

        foreach ( $fields_to_sanitize as $key => $sanitize_callback ) {
            if ( isset( $_POST[$key] ) ) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized dynamically via variable callback $sanitize_callback immediately below.
                $value = $sanitize_callback( wp_unslash( $_POST[$key] ) );
                update_post_meta( $post_id, '_' . $key, $value );
            }
        }

        if ( isset( $_POST['wpcm_scripts'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via custom method _sanitize_scripts_field which uses wp_kses.
            $sanitized_scripts = $this->_sanitize_scripts_field( wp_unslash( $_POST['wpcm_scripts'] ) );
            update_post_meta( $post_id, '_wpcm_scripts', $sanitized_scripts );
        }

        if ( class_exists( 'WPCM_Core' ) ) {
            WPCM_Core::clear_frontend_cache();
        }
    }

    public function clear_cache_on_delete( $post_id ) {
        if ( get_post_type( $post_id ) !== 'wpcm_service' ) {
            return;
        }

        if ( class_exists( 'WPCM_Core' ) ) {
            WPCM_Core::clear_frontend_cache();
        }
    }
}