<?php
/**
 * Hub Connector Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gerencia a conexão de Saída para o Hub e o Caching dos dados.
 * Esta classe é usada pelo plugin do Cliente.
 */
class WPCM_Hub_Connector {

    /**
     * Namespace da API do Hub (para conexões remotas).
     */
    protected static $hub_api_namespace = 'wpcm-hub/v1';

    /**
     * O nome do nosso cache (transient).
     */
    protected static $transient_key = 'wpcm_hub_data';

    /**
     * Busca os dados sincronizados do Hub (Configurações + Serviços).
     * Tenta primeiro pelo cache (transient). Se falhar, busca pela API.
     *
     * @return array|bool Os dados (settings, services, branding) ou false em falha.
     */
    public static function get_synced_data() {
        
        // 1. Tenta buscar do cache primeiro
        $cached_data = get_transient( self::$transient_key );
        if ( false !== $cached_data ) {
            return $cached_data;
        }

        // 2. Se não há cache, busca da API
        $options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $options ) ) {
            $options = array();
        }

        $hub_url = $options['agency_hub_url'] ?? '';
        $api_key = $options['agency_api_key'] ?? '';
        $status  = $options['agency_connection_status'] ?? 'disconnected';

        // Não tenta buscar se não estiver conectado
        if ( 'connected' !== $status || empty( $hub_url ) || empty( $api_key ) ) {
            return false;
        }

        $api_endpoint = trailingslashit( $hub_url ) . 'wp-json/' . self::$hub_api_namespace . '/settings';

        $response = wp_remote_get( $api_endpoint, array(
            'timeout'     => 15,
            'headers'     => array(
                'X-Api-Key' => $api_key, // Envia a chave no header
                'Accept'    => 'application/json',
            ),
            'user-agent'  => 'WPCM-Client/' . ( defined( 'WPCM_VERSION' ) ? WPCM_VERSION : '1.0' ),
        ) );

        if ( is_wp_error( $response ) ) {
            // A API falhou (ex: Hub offline)
            return false; 
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

        if ( $response_code !== 200 || ! is_array( $data ) || ! isset( $data['settings'] ) ) {
            // A API retornou um erro (ex: Chave inválida, 403)
            return false;
        }

        // 3. Sucesso! Salva no cache e retorna.
        // Cache de 12 horas.
        set_transient( self::$transient_key, $data, 12 * HOUR_IN_SECONDS );

        return $data;
    }

    /**
     * Força a limpeza do cache (transient).
     * Isso forçará uma nova busca na próxima vez que os dados forem necessários.
     */
    public static function clear_cache() {
        delete_transient( self::$transient_key );
    }
}