<?php
/**
 * Client API Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gerencia os Endpoints da REST API de entrada para o Plugin do Cliente.
 */
class WPCM_Client_API {

    protected $namespace = 'wpcm-client/v1';

    public function __construct() {
        add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
    }

    public function register_rest_routes() {
        
        register_rest_route( $this->namespace, '/clear_cache', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array( $this, 'handle_clear_cache' ),
            'permission_callback' => array( $this, 'permission_check' ),
        ) );

        register_rest_route( $this->namespace, '/test-connection', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array( $this, 'handle_test_connection' ),
            'permission_callback' => array( $this, 'permission_check' ),
        ) );
    }

    /**
     * Verifica as permissões da API.
     */
    public function permission_check( $request ) {
        
        $sent_key = $request->get_header( 'X-Api-Key' );

        if ( empty( $sent_key ) ) {
            $sent_key = $request->get_param( 'api_key' );
        }
        
        // FIX: Sanitização para garantir que não há caracteres estranhos
        $sent_key = sanitize_text_field( $sent_key );

        if ( empty( $sent_key ) ) {
            return new WP_Error( 'rest_forbidden', __( 'Missing API Key.', 'consent-manager-gdpr-ccpa' ), array( 'status' => 401 ) );
        }

        $options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $options ) || empty( $options['agency_api_key'] ) ) {
            return new WP_Error( 'rest_forbidden', __( 'Client site is not configured for Hub connection.', 'consent-manager-gdpr-ccpa' ), array( 'status' => 403 ) );
        }

        $stored_key = $options['agency_api_key'];

        if ( ! hash_equals( $stored_key, $sent_key ) ) {
             return new WP_Error( 'rest_forbidden', __( 'Invalid API Key.', 'consent-manager-gdpr-ccpa' ), array( 'status' => 403 ) );
        }

        return true;
    }

    public function handle_clear_cache( $request ) {
        
        if ( ! class_exists('WPCM_Hub_Connector') ) {
            return new WP_REST_Response( array(
                'success' => false,
                'error'   => 'Hub Connector class not found.',
            ), 500 );
        }

        WPCM_Hub_Connector::clear_cache();

        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Client cache cleared successfully.',
        ), 200 );
    }

    public function handle_test_connection( $request ) {
        return new WP_REST_Response( array(
            'success' => true,
            'message' => 'Connected.',
        ), 200 );
    }
}