<?php
/**
 * Setup Wizard Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPCM_Setup_Wizard {

    public function __construct() {
        add_action( 'admin_init', array( $this, 'redirect_to_wizard' ) );
        add_action( 'admin_menu', array( $this, 'register_wizard_page' ) );
        add_action( 'admin_post_wpcm_save_wizard', array( $this, 'save_wizard_settings' ) );
        add_action( 'admin_notices', array( $this, 'setup_incomplete_notice' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_wizard_assets' ) );
    }

    public function redirect_to_wizard() {
        if ( get_transient( 'wpcm_activation_redirect' ) ) {
            delete_transient( 'wpcm_activation_redirect' );

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Verificação de flag na URL
            if ( ! isset( $_GET['activate-multi'] ) ) {
                wp_safe_redirect( admin_url( 'admin.php?page=wpcm-setup-wizard' ) );
                exit;
            }
        }
    }

    public function register_wizard_page() {
        add_submenu_page(
            null,
            __( 'Setup Consent Manager', 'consent-manager-gdpr-ccpa' ),
            __( 'Setup Consent Manager', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'wpcm-setup-wizard',
            array( $this, 'render_wizard_page' )
        );
    }

    public function render_wizard_page() {
        if ( ! class_exists( 'WPCM_Service_Presets' ) ) {
            require_once WPCM_PLUGIN_PATH . 'includes/settings/class-service-presets.php';
        }
        $presets = WPCM_Service_Presets::get_all_presets();
        ?>
        <div class="wrap wpcm-wizard-wrap">
            <div class="wpcm-wizard-card">
                <div class="wpcm-wizard-logo">
                    <span class="dashicons dashicons-shield-alt" style="font-size: 48px; width: 48px; height: 48px; color: #4F46E5;"></span>
                </div>
                <h1><?php esc_html_e( 'Welcome to Consent Manager', 'consent-manager-gdpr-ccpa' ); ?></h1>
                <p class="wpcm-wizard-subtitle"><?php esc_html_e( 'Let\'s get your site compliant in less than a minute. Select the services you use below, and we\'ll configure them for you.', 'consent-manager-gdpr-ccpa' ); ?></p>

                <div class="wpcm-scanner-section" style="text-align: center; margin-bottom: 30px; padding: 20px; background: #f8f9fa; border: 1px dashed #ccc; border-radius: 5px;">
                    <h3 style="margin-top: 0;"><?php esc_html_e( 'Don\'t know which services you use?', 'consent-manager-gdpr-ccpa' ); ?></h3>
                    <p><?php esc_html_e( 'We can scan your homepage to detect Google, Meta, and other scripts automatically.', 'consent-manager-gdpr-ccpa' ); ?></p>
                    
                    <button type="button" id="wpcm-start-scan" class="button button-secondary">
                        <span class="dashicons dashicons-search" style="margin-top:4px;"></span> <?php esc_html_e( 'Auto-Scan Site', 'consent-manager-gdpr-ccpa' ); ?>
                    </button>
                    <span class="spinner" style="float:none;"></span>
                    
                    <div id="wpcm-scan-results" style="margin-top: 15px; text-align: left;"></div>
                </div>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <input type="hidden" name="action" value="wpcm_save_wizard">
                    <?php wp_nonce_field( 'wpcm_wizard_nonce', 'wpcm_wizard_security' ); ?>

                    <div class="wpcm-wizard-options">
                        <?php foreach ( $presets as $key => $preset ) : ?>
                            <label class="wpcm-wizard-option">
                                <input type="checkbox" name="wpcm_services[]" value="<?php echo esc_attr( $key ); ?>">
                                <div class="wpcm-wizard-option-content">
                                    <span class="wpcm-option-title"><?php echo esc_html( $preset['title'] ); ?></span>
                                    <span class="wpcm-option-desc"><?php echo esc_html( $preset['category'] ); ?></span>
                                </div>
                            </label>
                        <?php endforeach; ?>
                        
                        <label class="wpcm-wizard-option wpcm-option-manual">
                            <input type="checkbox" name="wpcm_manual_setup" value="1">
                            <div class="wpcm-wizard-option-content">
                                <span class="wpcm-option-title"><?php esc_html_e( 'None / Configure Manually later', 'consent-manager-gdpr-ccpa' ); ?></span>
                                <span class="wpcm-option-desc"><?php esc_html_e( 'I will add my services manually.', 'consent-manager-gdpr-ccpa' ); ?></span>
                            </div>
                        </label>
                    </div>

                    <div class="wpcm-wizard-footer">
                        <button type="submit" class="button button-primary button-hero"><?php esc_html_e( 'Finish Setup & Activate', 'consent-manager-gdpr-ccpa' ); ?></button>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=wpcm_settings' ) ); ?>" class="wpcm-skip-link"><?php esc_html_e( 'Skip this step', 'consent-manager-gdpr-ccpa' ); ?></a>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }

    public function save_wizard_settings() {
        check_admin_referer( 'wpcm_wizard_nonce', 'wpcm_wizard_security' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html( __( 'Sorry, you are not allowed to manage options for this site.', 'consent-manager-gdpr-ccpa' ) ) );
        }

        if ( ! class_exists( 'WPCM_Service_Presets' ) ) {
            require_once WPCM_PLUGIN_PATH . 'includes/settings/class-service-presets.php';
        }

        if ( empty( $_POST['wpcm_manual_setup'] ) && ! empty( $_POST['wpcm_services'] ) && is_array( $_POST['wpcm_services'] ) ) {
            $selected_services = array_map( 'sanitize_text_field', wp_unslash( $_POST['wpcm_services'] ) );
            WPCM_Service_Presets::create_services_from_presets( $selected_services );

            if ( in_array( 'youtube_vimeo', $selected_services ) ) {
                $options = get_option( 'wpcm_settings_options', array() );
                
                if ( ! is_array( $options ) ) {
                    $options = array();
                }

                $options['enable_iframe_blocker'] = '1';
                update_option( 'wpcm_settings_options', $options );
            }
        }

        update_option( 'wpcm_setup_complete', 1 );

        wp_safe_redirect( add_query_arg( 'wpcm-setup-success', '1', admin_url( 'admin.php?page=wpcm_settings' ) ) );
        exit;
    }

    public function setup_incomplete_notice() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $current_page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : '';
        
        if ( $current_page === 'wpcm-setup-wizard' ) {
            return;
        }

        if ( ! get_option( 'wpcm_setup_complete' ) ) {
            ?>
            <div class="notice notice-info is-dismissible wpcm-notice">
                <p>
                    <strong><?php esc_html_e( 'Consent Manager is almost ready.', 'consent-manager-gdpr-ccpa' ); ?></strong> 
                    <?php esc_html_e( 'To ensure compliance, please complete the quick setup wizard.', 'consent-manager-gdpr-ccpa' ); ?>
                </p>
                <p>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=wpcm-setup-wizard' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Run Setup Wizard', 'consent-manager-gdpr-ccpa' ); ?></a>
                </p>
            </div>
            <?php
        }
    }

    public function enqueue_wizard_assets( $hook ) {
        if ( ! is_string( $hook ) || strpos( $hook, 'wpcm-setup-wizard' ) === false ) {
            return;
        }
        
        wp_enqueue_style( 'wpcm-setup-wizard-style', WPCM_PLUGIN_URL . 'assets/css/setup-wizard.css', array(), WPCM_VERSION );
    }
}