<?php
/**
 * Consent Log Table Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

class WPCM_Consent_Log_Table extends WP_List_Table {

    public function __construct() {
        parent::__construct( array(
            'singular' => __( 'Consent Log', 'consent-manager-gdpr-ccpa' ),
            'plural'   => __( 'Consent Logs', 'consent-manager-gdpr-ccpa' ),
            'ajax'     => false
        ) );
    }

    public function get_columns() {
        return array(
            'user_hash'          => __( 'User (Hashed)', 'consent-manager-gdpr-ccpa' ),
            'ip_address'         => __( 'IP Address', 'consent-manager-gdpr-ccpa' ),
            'consent_categories' => __( 'Consents Given', 'consent-manager-gdpr-ccpa' ),
            'policy_version'     => __( 'Policy Version', 'consent-manager-gdpr-ccpa' ),
            'timestamp'          => __( 'Date & Time', 'consent-manager-gdpr-ccpa' )
        );
    }

    public function get_sortable_columns() {
        return array(
            'timestamp'      => array( 'timestamp', true ),
            'policy_version' => array( 'policy_version', false )
        );
    }

    protected function get_bulk_actions() {
        return array();
    }

    public function extra_tablenav( $which ) {
        if ( $which !== 'top' ) return;

        echo '<div class="alignleft actions">';
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Filtro de visualização, não requer nonce.
        $selected_date = isset( $_REQUEST['filter_date'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['filter_date'] ) ) : 'all';
        echo '<select name="filter_date" style="margin-right:5px;">';
        echo '<option value="all" ' . selected( $selected_date, 'all', false ) . '>' . esc_html( __( 'All Dates', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '<option value="today" ' . selected( $selected_date, 'today', false ) . '>' . esc_html( __( 'Today', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '<option value="7days" ' . selected( $selected_date, '7days', false ) . '>' . esc_html( __( 'Last 7 Days', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '<option value="30days" ' . selected( $selected_date, '30days', false ) . '>' . esc_html( __( 'Last 30 Days', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '<option value="thismonth" ' . selected( $selected_date, 'thismonth', false ) . '>' . esc_html( __( 'This Month', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '</select>';

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Filtro de visualização, não requer nonce.
        $selected_cat = isset( $_REQUEST['filter_category'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['filter_category'] ) ) : '';
        echo '<select name="filter_category" style="margin-right:5px;">';
        echo '<option value="">' . esc_html( __( 'All Consents', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '<option value="analytics" ' . selected( $selected_cat, 'analytics', false ) . '>' . esc_html( __( 'Accepted Analytics', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '<option value="marketing" ' . selected( $selected_cat, 'marketing', false ) . '>' . esc_html( __( 'Accepted Marketing', 'consent-manager-gdpr-ccpa' ) ) . '</option>';
        echo '</select>';

        submit_button( __( 'Filter', 'consent-manager-gdpr-ccpa' ), '', 'filter_action', false, array( 'id' => 'post-query-submit' ) );
        echo '</div>';

        ?>
        <div class="alignright actions">
            <a href="<?php echo esc_url( wp_nonce_url( add_query_arg( 'wpcm_action', 'export_csv', admin_url( 'admin.php?page=wpcm_consent_log' ) ), 'wpcm_export_csv_nonce', 'wpcm_export_nonce' ) ); ?>" class="button button-secondary" title="<?php esc_attr_e( 'Export current view to CSV', 'consent-manager-gdpr-ccpa' ); ?>">
                <span class="dashicons dashicons-download" style="margin-top: 4px;"></span> <?php esc_html_e( 'Export CSV', 'consent-manager-gdpr-ccpa' ); ?>
            </a>
        </div>
        <?php
    }

    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            
            case 'user_hash':
                $hash = esc_attr( $item[ $column_name ] );
                
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Leitura de parâmetro para construção de link.
                $current_page_slug = isset( $_REQUEST['page'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['page'] ) ) : 'wpcm_consent_log';
                
                $search_url = add_query_arg( array(
                    'page' => $current_page_slug,
                    's'    => $hash,
                ), admin_url( 'admin.php' ) );
                
                $short_hash = esc_html( substr( $hash, 0, 12 ) );

                return '<code title="' . $hash . '"><a href="' . esc_url( $search_url ) . '">' . $short_hash . '...</a></code> 
                        <span class="dashicons dashicons-admin-page wpcm-copy-icon" 
                              title="' . esc_attr__( 'Click to copy full hash', 'consent-manager-gdpr-ccpa' ) . '" 
                              data-hash="' . $hash . '"></span>';
            
            case 'ip_address':
                $ip = isset( $item[ $column_name ] ) ? esc_html( $item[ $column_name ] ) : '';
                if ( empty( $ip ) ) {
                    return '<em style="color:#999;">' . esc_html__( 'Not stored', 'consent-manager-gdpr-ccpa' ) . '</em>';
                }
                return '<code>' . $ip . '</code>';
            
            case 'consent_categories':
                $categories_string = $item[ $column_name ] ?? '';
                $categories = explode( ',', $categories_string );
                $badges = array();
                
                foreach ( array_filter( $categories ) as $cat_raw ) {
                    $cat = trim( $cat_raw );
                    
                    if ( strpos( $cat, 'service_' ) === 0 ) {
                        $service_id = intval( str_replace( 'service_', '', $cat ) );
                        
                        if ( $service_id > 0 ) {
                            $service_name = get_the_title( $service_id );
                            if ( $service_name ) {
                                $badges[] = '<span class="wpcm-badge wpcm-badge-service">' . esc_html( $service_name ) . '</span>';
                            }
                        }
                    } 
                    else {
                        $badges[] = '<span class="wpcm-badge wpcm-badge-' . esc_attr( $cat ) . '">' . esc_html( ucfirst( $cat ) ) . '</span>';
                    }
                }
                return implode( ' ', $badges );

            case 'policy_version':
                return '<span class="wpcm-version-tag">' . esc_html( $item[ $column_name ] ) . '</span>';
            
            case 'timestamp':
                $date = strtotime( $item[ $column_name ] );
                return date_i18n( get_option( 'date_format' ), $date ) . ' <span style="color:#999">' . date_i18n( get_option( 'time_format' ), $date ) . '</span>';
            
            default:
                return esc_html( $item[ $column_name ] ?? '' );
        }
    }

    public function prepare_items() {
        global $wpdb;
        $per_page = 20;
        $current_page = $this->get_pagenum();

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Filtros de URL.
        $search_term = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $filter_date = isset( $_REQUEST['filter_date'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['filter_date'] ) ) : 'all';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $filter_cat = isset( $_REQUEST['filter_category'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['filter_category'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $orderby_req = isset( $_GET['orderby'] ) ? sanitize_key( wp_unslash( $_GET['orderby'] ) ) : 'timestamp';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $order_req = isset( $_GET['order'] ) ? sanitize_key( wp_unslash( $_GET['order'] ) ) : 'DESC';

        $sortable_columns = $this->get_sortable_columns();
        $orderby = 'timestamp';
        if ( isset( $sortable_columns[ $orderby_req ] ) ) {
            $orderby = $orderby_req;
        }
        
        $order = 'DESC';
        if ( in_array( strtoupper( $order_req ), array( 'ASC', 'DESC' ) ) ) {
            $order = $order_req;
        }

        $cache_version = get_transient( 'wpcm_log_cache_v' );
        if ( false === $cache_version ) {
            $cache_version = time();
            set_transient( 'wpcm_log_cache_v', $cache_version, 12 * HOUR_IN_SECONDS );
        }

        $cache_params = array( $current_page, $per_page, $search_term, $filter_date, $filter_cat, $orderby, $order, $cache_version );
        $cache_key = 'wpcm_log_' . md5( implode( '_', $cache_params ) );
        $cache_group = 'wpcm_consent';
        
        $cached_data = wp_cache_get( $cache_key, $cache_group );

        if ( false === $cached_data ) {
            
            $sql_where_clauses = array( '%d = %d' );
            $sql_args = array( 1, 1 );

            if ( ! empty( $search_term ) ) {
                $sql_where_clauses[] = '( user_hash LIKE %s OR ip_address LIKE %s )';
                $sql_args[] = '%' . $wpdb->esc_like( $search_term ) . '%';
                $sql_args[] = '%' . $wpdb->esc_like( $search_term ) . '%';
            }

            if ( $filter_date !== 'all' ) {
                switch ( $filter_date ) {
                    case 'today': $sql_where_clauses[] = "DATE(timestamp) = CURDATE()"; break;
                    case '7days': $sql_where_clauses[] = "timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)"; $sql_args[] = 7; break;
                    case '30days': $sql_where_clauses[] = "timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)"; $sql_args[] = 30; break;
                    case 'thismonth': $sql_where_clauses[] = "MONTH(timestamp) = MONTH(CURRENT_DATE()) AND YEAR(timestamp) = YEAR(CURRENT_DATE())"; break;
                }
            }

            if ( ! empty( $filter_cat ) ) {
                $sql_where_clauses[] = 'consent_categories LIKE %s';
                $sql_args[] = '%' . $wpdb->esc_like( $filter_cat ) . '%';
            }

            $sql_where = ' WHERE ' . implode( ' AND ', $sql_where_clauses );
            $table_name = $wpdb->prefix . 'consent_log';

            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $total_items_query = "SELECT COUNT(id) FROM {$table_name} $sql_where";
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $total_items = $wpdb->get_var( $wpdb->prepare( $total_items_query, $sql_args ) );

            $offset = ( $current_page - 1 ) * $per_page;
            
            $sql_main_args = $sql_args;
            $sql_main_args[] = $per_page;
            $sql_main_args[] = $offset;
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $sql_main = "SELECT * FROM {$table_name} $sql_where ORDER BY $orderby $order LIMIT %d OFFSET %d";

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $items = $wpdb->get_results( $wpdb->prepare( $sql_main, $sql_main_args ), ARRAY_A );
            
            $cached_data = array(
                'items'       => $items,
                'total_items' => $total_items,
            );
            
            wp_cache_set( $cache_key, $cached_data, $cache_group, 600 );
        }

        $this->items = $cached_data['items'];
        $this->set_pagination_args( array(
            'total_items' => $cached_data['total_items'],
            'per_page'    => $per_page,
            'search'      => $search_term
        ) );
    }
}