<?php
/**
 * Admin Menu Class.
 *
 * @package Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WPCM_Admin_Menu {

    private $log_table;

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_plugin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'load_admin_assets' ) );
        add_action( 'admin_init', array( $this, 'handle_csv_export' ) );
        add_action( 'admin_post_wpcm_clear_hub_cache', array( $this, 'handle_clear_hub_cache' ) );
    }

    public function add_plugin_menu() {
        add_menu_page(
            __( 'Consent Manager', 'consent-manager-gdpr-ccpa' ),
            __( 'Consent Manager', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'wpcm_settings',
            array( $this, 'render_settings_page' ),
            'dashicons-shield-alt',
            80
        );
        
        add_submenu_page(
            'wpcm_settings',
            __( 'Settings', 'consent-manager-gdpr-ccpa' ),
            __( 'Settings', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'wpcm_settings',
            array( $this, 'render_settings_page' )
        );

        add_submenu_page(
            'wpcm_settings',
            __( 'Services', 'consent-manager-gdpr-ccpa' ),
            __( 'Services', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'edit.php?post_type=wpcm_service'
        );

        $hook_log = add_submenu_page(
            'wpcm_settings',
            __( 'Consent Log', 'consent-manager-gdpr-ccpa' ),
            __( 'Consent Log', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'wpcm_consent_log',
            array( $this, 'render_log_page' )
        );

        add_submenu_page(
            'wpcm_settings',
            __( 'Help & Guide', 'consent-manager-gdpr-ccpa' ),
            __( 'Help', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'wpcm_help',
            array( $this, 'render_help_page' )
        );

        add_submenu_page(
            'wpcm_settings',
            __( 'Status & Tools', 'consent-manager-gdpr-ccpa' ),
            __( 'Status & Tools', 'consent-manager-gdpr-ccpa' ),
            'manage_options',
            'wpcm_status',
            array( $this, 'render_status_page' )
        );

        add_action( "load-$hook_log", array( $this, 'instantiate_log_table' ) );
    }

    public function instantiate_log_table() {
        require_once WPCM_PLUGIN_PATH . 'includes/admin/class-consent-log-table.php';
        $this->log_table = new WPCM_Consent_Log_Table();
    }

    /**
     * Handles CSV Export with Batch Processing.
     */
    public function handle_csv_export() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'wpcm_consent_log' && 
             isset( $_GET['wpcm_action'] ) && $_GET['wpcm_action'] === 'export_csv' ) {

            if ( ! current_user_can( 'manage_options' ) ) { return; }

            check_admin_referer( 'wpcm_export_csv_nonce', 'wpcm_export_nonce' );

            global $wpdb;
            $table_name = $wpdb->prefix . 'consent_log';

            $filename = 'consent-log-' . gmdate( 'Y-m-d' ) . '.csv';
            header( 'Content-Type: text/csv; charset=utf-8' );
            header( 'Content-Disposition: attachment; filename=' . $filename );

            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
            $output = fopen( 'php://output', 'w' );
            
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv
            fputcsv( $output, array( 
                'Log ID', 
                'Date/Time (UTC)', 
                'User Hash', 
                'IP Address', 
                'Consented Categories', 
                'Policy Version',
                'Synced to Hub'
            ) );

            $batch_size = 1000; 
            $page = 0;

            while ( true ) {
                $offset = $page * $batch_size;
                
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
                $rows = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT * FROM {$table_name} ORDER BY id ASC LIMIT %d OFFSET %d",
                        $batch_size,
                        $offset
                    ),
                    ARRAY_A
                );
                // phpcs:enable

                if ( empty( $rows ) ) {
                    break;
                }

                foreach ( $rows as $row ) {
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv
                    fputcsv( $output, array(
                        $row['id'],
                        $row['timestamp'],
                        $row['user_hash'],
                        $row['ip_address'],
                        $row['consent_categories'],
                        $row['policy_version'],
                        $row['synced_to_hub']
                    ) );
                }
                
                unset( $rows ); 
                $page++; 
            }
            
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
            fclose( $output );
            exit;
        }
    }

    public function load_admin_assets( $hook_suffix ) {
        if ( ! is_string( $hook_suffix ) ) {
            return;
        }

        if ( strpos( $hook_suffix, 'wpcm_' ) !== false || strpos( $hook_suffix, 'wpcm-service' ) !== false ) {
            
            wp_enqueue_style( 'wp-color-picker' );
            wp_enqueue_style( 'wpcm-admin-style', WPCM_PLUGIN_URL . 'assets/css/admin.css', array(), WPCM_VERSION );
            
            wp_enqueue_script( 'wpcm-admin-script', WPCM_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery', 'wp-color-picker' ), WPCM_VERSION, true );

            wp_localize_script(
                'wpcm-admin-script',
                'wpcm_admin_vars',
                array(
                    'ajax_url'    => admin_url( 'admin-ajax.php' ),
                    'nonce'       => wp_create_nonce( 'wpcm_ajax_nonce' ),
                    'text_error'  => __( 'Error: AJAX request failed.', 'consent-manager-gdpr-ccpa' )
                )
            );
        }

        if ( strpos( $hook_suffix, 'wpcm-setup-wizard' ) !== false ) {
            
            wp_enqueue_script( 'wpcm-scanner', WPCM_PLUGIN_URL . 'assets/js/admin-scanner.js', array( 'jquery' ), WPCM_VERSION, true );
            
            if ( ! class_exists( 'WPCM_Service_Presets' ) ) {
                require_once WPCM_PLUGIN_PATH . 'includes/settings/class-service-presets.php';
            }
            
            $signatures = array();
            if ( method_exists( 'WPCM_Service_Presets', 'get_scanner_signatures' ) ) {
                $signatures = WPCM_Service_Presets::get_scanner_signatures();
            }

            wp_localize_script( 'wpcm-scanner', 'wpcm_scanner_vars', array(
                'home_url'   => home_url(),
                'admin_url'  => admin_url(),
                'nonce'      => wp_create_nonce( 'wpcm_wizard_nonce' ), 
                'signatures' => $signatures
            ));
        }
    }

    private function render_stats_dashboard() {
        global $wpdb;
        
        $filter_date = 'all';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_REQUEST['filter_date'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $filter_date = sanitize_text_field( wp_unslash( $_REQUEST['filter_date'] ) );
        }
        
        $cache_key = 'wpcm_stats_' . $filter_date;
        $stats = wp_cache_get( $cache_key, 'wpcm_consent' );

        if ( false === $stats ) {
            
            $table_name = $wpdb->prefix . 'consent_log';
            
            $sql_conditions = array( "%d = %d" );
            $sql_args = array( 1, 1 );

            if ( $filter_date !== 'all' ) {
                switch ( $filter_date ) {
                    case 'today': 
                        $sql_conditions[] = "DATE(timestamp) = CURDATE()"; 
                        break;
                    case '7days': 
                        $sql_conditions[] = "timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)"; 
                        $sql_args[] = 7;
                        break;
                    case '30days': 
                        $sql_conditions[] = "timestamp >= DATE_SUB(NOW(), INTERVAL %d DAY)"; 
                        $sql_args[] = 30;
                        break;
                    case 'thismonth': 
                        $sql_conditions[] = "MONTH(timestamp) = MONTH(CURRENT_DATE()) AND YEAR(timestamp) = YEAR(CURRENT_DATE())"; 
                        break;
                }
            }
            
            // Usamos block disable para silenciar erros de SQL complexo (Wildcards, Interpolation, etc)
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.LikeWildcardsInQuery, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $stats = $wpdb->get_row( 
                $wpdb->prepare( 
                    "SELECT COUNT(id) as total, SUM(CASE WHEN consent_categories LIKE '%%analytics%%' AND consent_categories LIKE '%%marketing%%' THEN 1 ELSE 0 END) as full_consent, SUM(CASE WHEN consent_categories = 'necessary' THEN 1 ELSE 0 END) as necessary_only FROM {$table_name} WHERE " . implode( ' AND ', $sql_conditions ), 
                    $sql_args 
                ), 
                ARRAY_A 
            );
            // phpcs:enable
            
            wp_cache_set( $cache_key, $stats, 'wpcm_consent', 3600 );
        }
        
        $total = (int) $stats['total']; $full = (int) $stats['full_consent']; $necessary = (int) $stats['necessary_only']; $partial = max( 0, $total - $full - $necessary ); $full_rate = $total > 0 ? round( ($full / $total) * 100 ) : 0;
        
        ?>
        <div class="wpcm-dashboard-stats">
            <div class="wpcm-stat-card wpcm-border-blue"><div class="wpcm-stat-icon dashicons dashicons-chart-area"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $total ) ); ?></span><span class="wpcm-stat-label"><?php esc_html_e( 'Total Interactions', 'consent-manager-gdpr-ccpa' ); ?></span></div></div>
            <div class="wpcm-stat-card wpcm-border-green"><div class="wpcm-stat-icon dashicons dashicons-yes-alt"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $full ) ); ?> <small>(<?php echo esc_html( $full_rate ); ?>%)</small></span><span class="wpcm-stat-label"><?php esc_html_e( 'Accepted All', 'consent-manager-gdpr-ccpa' ); ?></span></div></div>
            <div class="wpcm-stat-card wpcm-border-orange"><div class="wpcm-stat-icon dashicons dashicons-filter"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $partial ) ); ?></span><span class="wpcm-stat-label"><?php esc_html_e( 'Partially Accepted', 'consent-manager-gdpr-ccpa' ); ?></span></div></div>
            <div class="wpcm-stat-card wpcm-border-red"><div class="wpcm-stat-icon dashicons dashicons-dismiss"></div><div class="wpcm-stat-content"><span class="wpcm-stat-number"><?php echo esc_html( number_format_i18n( $necessary ) ); ?></span><span class="wpcm-stat-label"><?php esc_html_e( 'Rejected (Necessary)', 'consent-manager-gdpr-ccpa' ); ?></span></div></div>
        </div>
        <?php
    }

    public function render_settings_page() {
        global $wp_settings_sections;
        $active_tab = 'general';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['tab'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $active_tab = sanitize_key( wp_unslash( $_GET['tab'] ) );
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <h2 class="nav-tab-wrapper">
                <a href="?page=wpcm_settings&tab=general" class="nav-tab <?php echo $active_tab == 'general' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'General', 'consent-manager-gdpr-ccpa' ); ?></a>
                <a href="?page=wpcm_settings&tab=design" class="nav-tab <?php echo $active_tab == 'design' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Design', 'consent-manager-gdpr-ccpa' ); ?></a>
                <a href="?page=wpcm_settings&tab=content" class="nav-tab <?php echo $active_tab == 'content' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Content', 'consent-manager-gdpr-ccpa' ); ?></a>
                
                <a href="?page=wpcm_settings&tab=agency" class="nav-tab <?php echo $active_tab == 'agency' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Agency Hub', 'consent-manager-gdpr-ccpa' ); ?></a>
            </h2>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'wpcm_settings_group' );
                $page_slug = 'wpcm_settings_page_' . $active_tab;
                
                if ( ! isset( $wp_settings_sections[$page_slug] ) && $active_tab !== 'agency' ) {
                    $page_slug = 'wpcm_settings_page_general';
                }

                if ( isset( $wp_settings_sections[$page_slug] ) ) {
                    do_settings_sections( $page_slug );
                }
                
                submit_button( __( 'Save Changes', 'consent-manager-gdpr-ccpa' ) );
                ?>
            </form>
        </div>
        <?php
    }

    public function render_log_page() {
        $this->log_table->prepare_items();
        
        $page_attr = '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_REQUEST['page'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $page_attr = sanitize_text_field( wp_unslash( $_REQUEST['page'] ) );
        }
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e( 'Consent Log', 'consent-manager-gdpr-ccpa' ); ?></h1>
            <?php $this->render_stats_dashboard(); ?>
            <hr class="wp-header-end">
            <form id="consent-log-form" method="get">
                <input type="hidden" name="page" value="<?php echo esc_attr( $page_attr ); ?>" />
                <?php
                $this->log_table->search_box( __( 'Search Hash', 'consent-manager-gdpr-ccpa' ), 'wpcm_search_hash' );
                $this->log_table->display();
                ?>
            </form>
        </div>
        <?php
    }
    
    public function render_help_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Consent Manager - Help Center', 'consent-manager-gdpr-ccpa' ); ?></h1>
            <p><?php esc_html_e( 'Documentation and help content here.', 'consent-manager-gdpr-ccpa' ); ?></p>
        </div>
        <?php
    }
    
    public function handle_clear_hub_cache() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Acesso negado.' );
        }

        check_admin_referer( 'wpcm_clear_cache_nonce', 'wpcm_clear_cache_field' );

        if ( class_exists( 'WPCM_Hub_Connector' ) ) {
            WPCM_Hub_Connector::clear_cache();
        }
        
        $redirect_url = add_query_arg(
            array(
                'page' => 'wpcm_status',
                'cache_cleared' => '1',
            ),
            admin_url( 'admin.php' )
        );
        
        wp_safe_redirect( $redirect_url );
        exit;
    }

    public function render_status_page() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if ( isset( $_GET['cache_cleared'] ) && $_GET['cache_cleared'] === '1' ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Hub cache successfully cleared!', 'consent-manager-gdpr-ccpa' ) . '</p></div>';
        }
        
        $options = get_option( 'wpcm_settings_options' );
        if ( ! is_array( $options ) ) $options = array();
        
        $status    = $options['agency_connection_status'] ?? 'disconnected';
        $hub_url   = $options['agency_hub_url'] ?? __( 'Not configured', 'consent-manager-gdpr-ccpa' );
        $hub_data  = get_transient( 'wpcm_hub_data' );

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Status & Tools', 'consent-manager-gdpr-ccpa' ); ?></h1>
            
            <div style="max-width: 800px; margin-top: 25px;">
                
                <div class="card" style="padding: 20px; margin-bottom: 20px; background: #fff; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <h2><?php esc_html_e( 'Hub Connection Status', 'consent-manager-gdpr-ccpa' ); ?></h2>
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Connection Status', 'consent-manager-gdpr-ccpa' ); ?></th>
                                <td>
                                    <?php if ( $status === 'connected' ) : ?>
                                        <span style="color: #10b981; font-weight: 600;"><?php esc_html_e( 'Connected', 'consent-manager-gdpr-ccpa' ); ?></span>
                                    <?php else : ?>
                                        <span style="color: #ef4444; font-weight: 600;"><?php esc_html_e( 'Disconnected', 'consent-manager-gdpr-ccpa' ); ?></span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Configured Hub URL', 'consent-manager-gdpr-ccpa' ); ?></th>
                                <td><code><?php echo esc_html( $hub_url ); ?></code></td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e( 'Cache Status', 'consent-manager-gdpr-ccpa' ); ?></th>
                                <td>
                                    <?php if ( false !== $hub_data ) : ?>
                                        <span style="color: #10b981;"><?php esc_html_e( 'Data is currently cached.', 'consent-manager-gdpr-ccpa' ); ?></span>
                                    <?php else : ?>
                                        <span style="color: #f59e0b;"><?php esc_html_e( 'No data in cache. Will fetch on next front-end load.', 'consent-manager-gdpr-ccpa' ); ?></span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <div class="card" style="padding: 20px; margin-bottom: 20px; background: #fff; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <h2><?php esc_html_e( 'Tools', 'consent-manager-gdpr-ccpa' ); ?></h2>
                    <p><?php esc_html_e( 'If you made changes in your Agency Hub but don\'t see them reflected here, use this button to force the plugin to fetch new data.', 'consent-manager-gdpr-ccpa' ); ?></p>
                    
                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                        <input type="hidden" name="action" value="wpcm_clear_hub_cache">
                        <?php wp_nonce_field( 'wpcm_clear_cache_nonce', 'wpcm_clear_cache_field' ); ?>
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e( 'Clear Hub Cache & Force Sync', 'consent-manager-gdpr-ccpa' ); ?>
                        </button>
                    </form>
                </div>

            </div>
        </div>
        <?php
    }
}