<?php
/**
 * Plugin Name:       Consent Manager - GDPR & CCPA
 * Plugin URI:        https://wpconsentmanage.com/wp-con
 * Description:       A complete, self-hosted solution for GDPR/LGPD cookie consent and true script blocking.
 * Version:           1.1.9
 * Requires at least: 5.6
 * Requires PHP:      7.4
 * Author:            pitterlopes
 * Author URI:        https://wpconsentmanage.com/wp-consent-manager/
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       consent-manager-gdpr-ccpa
 * Domain Path:       /languages
 *
 * @package           Consent_Manager_GDPR_CCPA
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; 
}

define( 'WPCM_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'WPCM_VERSION', '1.1.8' );
define( 'WPCM_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'WPCM_PLUGIN_URL', plugin_dir_url( __FILE__ ) );


// --- Carrega os componentes essenciais que precisam rodar ANTES do 'init' ---
// 1. O Conector do Hub
require_once WPCM_PLUGIN_PATH . 'includes/api/class-hub-connector.php';
// 2. A classe de White-Label
require_once WPCM_PLUGIN_PATH . 'includes/core/class-whitelabel.php';
// 3. A classe de CPT (Custom Post Type)
require_once WPCM_PLUGIN_PATH . 'includes/core/class-cpt-services.php';

// Instancia as classes que rodam globalmente
new WPCM_White_Label();
new WPCM_CPT_Services();


if ( ! class_exists( 'WPCM_Core' ) ) {
    
    /**
     * Classe principal
     * * @package Consent_Manager_GDPR_CCPA
     */
    class WPCM_Core {
        
        public function __construct() {
            $this->init_components();
            add_action( 'wpcm_daily_log_cleanup', array( $this, 'run_log_cleanup' ) );
            add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
        }

        /**
         * Carrega todos os outros componentes no hook 'init'.
         */
        public function init_components() {
            
            // Settings
            require_once WPCM_PLUGIN_PATH . 'includes/settings/class-settings-api.php';
            
            // Admin
            require_once WPCM_PLUGIN_PATH . 'includes/admin/class-admin-menu.php';
            
            // Public
            require_once WPCM_PLUGIN_PATH . 'includes/public/class-public.php';
            
            // Database / AJAX
            require_once WPCM_PLUGIN_PATH . 'includes/database/class-ajax-handler.php';
            
            // API
            require_once WPCM_PLUGIN_PATH . 'includes/api/class-client-api.php';
            
            // Jobs
            require_once WPCM_PLUGIN_PATH . 'includes/jobs/class-log-sync-scheduler.php';
            
            if ( is_admin() ) {
                // Admin
                require_once WPCM_PLUGIN_PATH . 'includes/admin/class-consent-log-table.php';
                require_once WPCM_PLUGIN_PATH . 'includes/admin/class-setup-wizard.php';
                
                // Settings
                require_once WPCM_PLUGIN_PATH . 'includes/settings/class-service-presets.php';
            }

            new WPCM_Settings_API();
            new WPCM_Admin_Menu();
            new WPCM_Public();
            new WPCM_Ajax_Handler();
            new WPCM_Client_API();
            new WPCM_Log_Sync_Scheduler();

            if ( is_admin() ) {
                 new WPCM_Setup_Wizard();
            }
        }

        public static function activate() {
            global $wpdb;
            $table_name = $wpdb->prefix . 'consent_log';
            $charset_collate = $wpdb->get_charset_collate();
            
            // Adicionamos a coluna 'ip_address' na definição da tabela.
            $sql = "CREATE TABLE $table_name (
                id BIGINT(20) NOT NULL AUTO_INCREMENT,
                user_hash VARCHAR(64) NOT NULL,
                timestamp DATETIME NOT NULL,
                consent_categories VARCHAR(255) NOT NULL,
                policy_version VARCHAR(20) NOT NULL,
                ip_address VARCHAR(45) NOT NULL DEFAULT '',
                synced_to_hub TINYINT(1) NOT NULL DEFAULT 0,
                PRIMARY KEY (id),
                INDEX idx_synced_to_hub (synced_to_hub)
            ) $charset_collate;";
            
            require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
            dbDelta( $sql );
            
            if ( ! wp_next_scheduled( 'wpcm_daily_log_cleanup' ) ) {
                wp_schedule_event( time(), 'daily', 'wpcm_daily_log_cleanup' );
            }
            
            if ( ! class_exists('WPCM_Log_Sync_Scheduler') ) {
                require_once WPCM_PLUGIN_PATH . 'includes/jobs/class-log-sync-scheduler.php';
            }
            WPCM_Log_Sync_Scheduler::schedule_cron_event();

            set_transient( 'wpcm_activation_redirect', true, 60 );
            add_option( 'wpcm_version', WPCM_VERSION );
            flush_rewrite_rules();
        }

        public static function deactivate() {
            wp_clear_scheduled_hook( 'wpcm_daily_log_cleanup' );
            
            if ( ! class_exists('WPCM_Log_Sync_Scheduler') ) {
                require_once WPCM_PLUGIN_PATH . 'includes/jobs/class-log-sync-scheduler.php';
            }
            WPCM_Log_Sync_Scheduler::clear_cron_event();

            delete_transient( 'wpcm_activation_redirect' );
            flush_rewrite_rules();
        }

        public function run_log_cleanup() {
            $options = get_option( 'wpcm_settings_options' );
            if ( !is_array($options) ) $options = array(); 
            
            $period = $options['log_retention_period'] ?? '12';
            if ( empty( $period ) || $period === '0' ) return;
            global $wpdb;
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $deleted_rows = $wpdb->query( 
                $wpdb->prepare( 
                    "DELETE FROM {$wpdb->prefix}consent_log WHERE timestamp < DATE_SUB( NOW(), INTERVAL %d MONTH )", 
                    intval( $period ) 
                ) 
            );

            if ( $deleted_rows > 0 ) {
                self::clear_log_cache();
            }
        }

        public static function clear_frontend_cache() {
            wp_cache_delete( 'wpcm_cats_with_scripts', 'wpcm_consent' );
            wp_cache_delete( 'wpcm_cats_no_scripts', 'wpcm_consent' );
            if ( class_exists('WPCM_Hub_Connector') ) {
                WPCM_Hub_Connector::clear_cache();
            }
        }

        public static function clear_log_cache() {
            $stats_keys = array( 'wpcm_stats_all', 'wpcm_stats_today', 'wpcm_stats_7days', 'wpcm_stats_30days', 'wpcm_stats_thismonth' );
            foreach ( $stats_keys as $key ) {
                wp_cache_delete( $key, 'wpcm_consent' );
            }
            set_transient( 'wpcm_log_cache_v', time() ); 
        }

        public function load_textdomain() {
            // phpcs:ignore PluginCheck.CodeAnalysis.DiscouragedFunctions.load_plugin_textdomainFound
            load_plugin_textdomain(
                'consent-manager-gdpr-ccpa',
                false,
                dirname( WPCM_PLUGIN_BASENAME ) . '/languages/'
            );
        }

    } // Fim da WPCM_Core
} // Fim do if ! class_exists

register_activation_hook( __FILE__, array( 'WPCM_Core', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'WPCM_Core', 'deactivate' ) );

function wpcm_init_plugin() {
    
    // Define as opções padrão na ativação
    if ( ! get_option( 'wpcm_settings_options' ) ) {
        if ( ! class_exists( 'WPCM_Settings_API' ) ) {
            require_once WPCM_PLUGIN_PATH . 'includes/settings/class-settings-api.php';
        }
        $default_options = WPCM_Settings_API::get_default_options();
        $default_options['policy_page_url'] = get_privacy_policy_url();
        add_option( 'wpcm_settings_options', $default_options );
    }
    
    // Garante que o cron esteja agendado caso a ativação falhe
    if ( class_exists('WPCM_Log_Sync_Scheduler') ) {
        WPCM_Log_Sync_Scheduler::schedule_cron_event();
    }
    
    new WPCM_Core();
}
add_action( 'init', 'wpcm_init_plugin' );