/**
 * Public Frontend Scripts.
 *
 * @package Consent_Manager_GDPR_CCPA
 */
(function($) {
    'use strict';

    $(document).ready(function() {
        
        // Verificação de segurança para variáveis
        if (typeof wpcm_vars_footer === 'undefined') { return; }

        var cookieName = wpcm_vars_footer.cookie.name;
        var policyVersion = wpcm_vars_footer.cookie.version;
        // NOVO: Verifica se o Advanced Mode está ativo
        var gcmEnabled = wpcm_vars_footer.gcm_enabled === true;
        var allConsentIDs = [];

        if (wpcm_vars_footer.categories) {
            wpcm_vars_footer.categories.forEach(function(cat) {
                allConsentIDs.push(cat.id);
                if (cat.services) {
                    cat.services.forEach(function(service) {
                        allConsentIDs.push(service.id);
                    });
                }
            });
        }

        /**
         * NOVO: Atualiza o Google Consent Mode v2 via API
         */
        function updateGoogleConsent(consentedIDs) {
            if (!gcmEnabled || typeof gtag !== 'function') return;

            // Mapeamento padrão: assumimos 'denied'
            var consentUpdate = {
                'ad_storage': 'denied',
                'ad_user_data': 'denied',
                'ad_personalization': 'denied',
                'analytics_storage': 'denied'
            };

            // Verifica se 'marketing' foi aceito
            if (consentedIDs.includes('marketing')) {
                consentUpdate.ad_storage = 'granted';
                consentUpdate.ad_user_data = 'granted';
                consentUpdate.ad_personalization = 'granted';
            }

            // Verifica se 'analytics' foi aceito
            if (consentedIDs.includes('analytics')) {
                consentUpdate.analytics_storage = 'granted';
            }

            // Envia atualização para o Google
            gtag('consent', 'update', consentUpdate);
            // Evento para Tag Manager disparar tags baseadas na atualização
            if (typeof window.dataLayer !== 'undefined') {
                window.dataLayer.push({'event': 'wpcm_consent_update'});
            }
        }

        /**
         * *** AQUI ESTÁ A MUDANÇA (SPRINT 17) ***
         * A função foi reordenada para ser "inquebrável".
         * O feedback visual (fechar o banner) e o salvamento do cookie/log
         * agora acontecem ANTES da execução dos scripts de terceiros.
         */
        function saveConsent(consentedIDs) {
            var consentData = { version: policyVersion, categories: consentedIDs };

            // 1. Ação mais importante: Feedback visual IMEDIATO.
            $('#wpcm-consent-banner, #wpcm-modal-overlay').fadeOut(300);

            // 2. Salva o cookie.
            if (typeof window.wpcm_set_cookie === 'function') {
                window.wpcm_set_cookie(cookieName, JSON.stringify(consentData), 365);
            }
            
            // 3. Envia o log para o banco de dados.
            $.post(wpcm_vars_footer.ajax_url, {
                action: 'wpcm_log_consent',
                nonce: wpcm_vars_footer.nonce,
                consent_data: JSON.stringify(consentData)
            });

            // 4. Atualiza Google Consent Mode v2 (IMEDIATAMENTE)
            updateGoogleConsent(consentedIDs);

            // 5. Roda a lógica de script (que pode quebrar) POR ÚLTIMO.
            setTimeout(function() {
                if (typeof window.wpcm_run_consent_logic === 'function') {
                    window.wpcm_run_consent_logic(consentedIDs);
                }
            }, 1); 
        }

        function restoreModalState(consentedIDs) {
            $('.wpcm-category-checkbox:not(:disabled), .wpcm-service-checkbox').prop('checked', false);
            consentedIDs.forEach(function(id) {
                $('.wpcm-category-checkbox[data-category-id="' + id + '"]').prop('checked', true);
                $('.wpcm-service-checkbox[data-service-id="' + id + '"]').prop('checked', true);
            });
        }

        // --- LISTENERS ---
        $(document).on('click', '.wpcm-category-expand', function(e) {
            e.preventDefault();
            e.stopPropagation(); // CRUCIAL para o acordeão funcionar bem
            var $list = $(this).closest('.wpcm-category-row').find('.wpcm-services-list');
            if ($list.is(':visible')) {
                $list.slideUp(200);
                $(this).attr('aria-expanded', 'false');
            } else {
                $list.slideDown(200);
                $(this).attr('aria-expanded', 'true');
            }
        });

        $(document).on('click', '#wpcm-customize', function() {
            var c = window.wpcm_get_cookie(cookieName);
            if (c) { try { var d = JSON.parse(c); if (d.categories) restoreModalState(d.categories); } catch(e){} }
            $('#wpcm-modal-overlay').fadeIn(200);
        });

        $(document).on('change', '.wpcm-category-checkbox', function() {
            $('.wpcm-service-checkbox[data-parent-category="' + $(this).data('category-id') + '"]').prop('checked', $(this).is(':checked')).trigger('change');
        });
        $(document).on('change', '.wpcm-service-checkbox', function() {
             if ($(this).is(':checked')) $('.wpcm-category-checkbox[data-category-id="' + $(this).data('parent-category') + '"]').prop('checked', true);
        });

        $('#wpcm-accept').on('click', function() { saveConsent(allConsentIDs); });
        $('#wpcm-reject').on('click', function() { saveConsent(['necessary']); });
        $('#wpcm-save-prefs').on('click', function() {
            var s = [];
            $('.wpcm-category-checkbox:checked').each(function() { s.push($(this).data('category-id')); });
            $('.wpcm-service-checkbox:checked').each(function() { s.push($(this).data('service-id')); });
            saveConsent(s);
        });
        $(document).on('click', '#wpcm-modal-close, #wpcm-modal-overlay', function(e) {
            if (e.target.id === 'wpcm-modal-overlay' || e.target.id === 'wpcm-modal-close') $('#wpcm-modal-overlay').fadeOut(200);
        });

        if (!window.wpcm_consent_given) $('#wpcm-consent-banner').fadeIn(300);
    });
})(jQuery);